/**
* ESPromise.ts
* @author  Nev Wylie (newylie))
* @copyright Microsoft 2019
* Simplified wrapper to provide ES6 style Promise callback handling for older browsers
*/
/**
 * This defines the handler function for when a promise is resolved.
 * @param value This is the value passed as part of resolving the Promise
 * @return This may return a value, another Promise or void. See [[then]] for how the value is handled.
 */
export type ESPromiseOnResolvedFunc<T> = (value: T) => T | ESPromise<T | any> | void;
/**
 * This defines the handler function for when a promise is rejected.
 * @param value This is the value passed as part of resolving the Promise
 * @return This may return a value, another Promise or void. @see then for how the value is handled.
 */
export type ESPromiseOnRejectedFunc<T> = (reason: any) => T | ESPromise<T | any> | void;
/**
 * Defines the signature of the resolve function passed to the resolverFunc (in the Promise constructor)
 * @param value The value to resolve the Promise with
 * @returns Nothing
 */
export type ResolverResolveFunc<T> = (value?: T | void) => void;
/**
 * Defines the signature of the reject function passed to the resolverFunc (in the Promise constructor)
 * @param value The value to reject the Promise with
 * @returns Nothing
 */
export type ResolverRejectFunc<T> = (reason?: T | Error | void) => void;
/**
 * Simplified wrapper to provide ES6 style Promise callback handling for older browsers
 */
export default class ESPromise<T> {
    /**
     * The Promise.resolve() method returns a Promise object that is resolved with a given value. If the value is a promise, that promise is returned;
     * if the value is a thenable (i.e. has a "then" method), the returned promise will "follow" that thenable, adopting its eventual state; otherwise
     * the returned promise will be fulfilled with the value. This function flattens nested layers of promise-like objects (e.g. a promise that resolves
     * to a promise that resolves to something) into a single layer.
     * @param value Argument to be resolved by this Promise. Can also be a Promise or a thenable to resolve.
     */
    static resolve<T>(value?: any): ESPromise<any>;
    /**
     * The Promise.reject() method returns a Promise object that is rejected with a given reason.
     * @param reason The reason why this Promise rejected.
     */
    static reject<T>(reason?: T | Error | void): ESPromise<T>;
    /**
     * The Promise.all() method returns a single Promise that resolves when all of the promises passed as an iterable
     * have resolved or when the iterable contains no promises. It rejects with the reason of the first promise that
     * rejects. There is no implied ordering in the execution of the array of Promises given. On some computers, they
     * may be executed in parallel, or in some sense concurrently, while on others they may be executed serially. For
     * this reason, there must be no dependency in any Promise on the order of execution of the Promises.
     * This method can be useful for aggregating the results of multiple promises.
     * FulfillmentSection - The returned promise is fulfilled with an array containing all the values of the iterable
     * passed as argument (also non-promise values).
     * If an empty iterable is passed, then this method returns (synchronously) an already resolved promise.
     * If all of the passed-in promises fulfill, or are not promises, the promise returned by Promise.all is fulfilled
     * asynchronously.
     * RejectionSection - If any of the passed-in promises reject, Promise.all asynchronously rejects with the value of
     * the promise that rejected, whether or not the other promises have resolved.
     * @param iterable
     */
    static all<T>(iterable: any[]): ESPromise<T>;
    /**
     * The race function returns a Promise that is settled the same way (and takes the same value) as the first promise
     * that settles amongst the promises of the iterable passed as an argument.
     * If the iterable passed is empty, the promise returned will be forever pending.
     * If the iterable contains one or more non-promise value and/or an already settled promise, then Promise.race will
     * resolve to the first of these values found in the iterable.
     * @param iterable
     */
    static race<T>(iterable: any[]): ESPromise<T>;
    /**
     * The Promise object represents the eventual completion (or failure) of an asynchronous operation, and its resulting value.
     * @param resolverFunc A function that is passed with the arguments resolve and reject. The executor function is executed
     * immediately by the Promise implementation, passing resolve and reject functions (the executor is called before the Promise
     * constructor even returns the created object). The resolve and reject functions, when called, resolve or reject the promise,
     * respectively. The executor normally initiates some asynchronous work, and then, once that completes, either calls the resolve
     * function to resolve the promise or else rejects it if an error occurred. If an error is thrown in the executor function, the
     * promise is rejected. The return value of the executor is ignored.
     */
    constructor(resolverFunc: (resolve: ResolverResolveFunc<T>, reject: ResolverRejectFunc<T>) => void);
    /**
     * The then() method returns a Promise. It takes up to two arguments: callback functions for the success and failure cases of the Promise.
     * @param onResolved A Function called if the Promise is fulfilled. This function has one argument, the fulfillment value. If it is not a
     * function, it is internally replaced with an "Identity" function (it returns the received argument).
     * @param onRejected A Function called if the Promise is rejected. This function has one argument, the rejection reason. If it is not a
     * function, it is internally replaced with a "Thrower" function (it throws an error it received as argument).
     * @returns Once a Promise is fulfilled or rejected, the respective handler function (onFulfilled or onRejected) will be called asynchronously
     * (scheduled in the current thread loop). The behavior of the handler function follows a specific set of rules. If a handler function:
     * - returns a value, the promise returned by then gets resolved with the returned value as its value;
     * - doesn't return anything, the promise returned by then gets resolved with an undefined value;
     * - throws an error, the promise returned by then gets rejected with the thrown error as its value;
     * - returns an already fulfilled promise, the promise returned by then gets fulfilled with that promise's value as its value;
     * - returns an already rejected promise, the promise returned by then gets rejected with that promise's value as its value;
     * - returns another pending promise object, the resolution/rejection of the promise returned by then will be subsequent to the
     * resolution/rejection of the promise returned by the handler. Also, the value of the promise returned by then will be the same as the value of the promise returned by the handler.
     */
    then(onResolved: ESPromiseOnResolvedFunc<T>, onRejected?: ESPromiseOnRejectedFunc<T>): ESPromise<T>;
    /**
     * The catch() method returns a Promise and deals with rejected cases only. It behaves the same as calling Promise.prototype.then(undefined, onRejected)
     * (in fact, calling obj.catch(onRejected) internally calls obj.then(undefined, onRejected)). This means that you have to provide an onRejected function
     * even if you want to fall back to an undefined result value - for example obj.catch(() => {}).
     * @param onRejected A Function called when the Promise is rejected. This function has one argument: reason The rejection reason.
     * @returns Internally calls Promise.prototype.then on the object upon which it was called, passing the parameters undefined and the received
     * onRejected handler. Returns the value of that call, which is a Promise.
     */
    catch(onRejected: ESPromiseOnRejectedFunc<T>): ESPromise<T>;
}
