"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimulatedRoomTakedownCapability = exports.SynapseAdminRoomTakedownCapability = exports.SynapseAdminRoomDetailsProvider = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const typescript_result_1 = require("@gnuxie/typescript-result");
require("../RoomTakedownCapability"); // needed for the interface to load.
const log = new matrix_protection_suite_1.Logger("SynapseAdminRoomTakedownCapability");
class SynapseAdminRoomDetailsProvider {
    constructor(adminClient) {
        this.adminClient = adminClient;
        // nothing to do mare.
    }
    async getRoomDetails(roomID) {
        const detailsResponse = await this.adminClient.getRoomDetails(roomID);
        if ((0, typescript_result_1.isError)(detailsResponse)) {
            return detailsResponse;
        }
        else {
            return (0, typescript_result_1.Ok)({
                name: detailsResponse.ok?.name ?? undefined,
                creator: detailsResponse.ok?.creator ?? undefined,
                avatar: detailsResponse.ok?.avatar ?? undefined,
                topic: detailsResponse.ok?.topic ?? undefined,
                joined_members: detailsResponse.ok?.joined_members ?? undefined,
                room_id: roomID,
            });
        }
    }
}
exports.SynapseAdminRoomDetailsProvider = SynapseAdminRoomDetailsProvider;
class SynapseAdminRoomTakedownCapability {
    constructor(adminClient) {
        this.adminClient = adminClient;
        this.requiredPermissions = [];
        this.requiredStatePermissions = [];
        this.requiredEventPermissions = [];
        this.roomDetailsProvider = new SynapseAdminRoomDetailsProvider(this.adminClient);
        // nothing to do mare.
    }
    async isRoomTakendown(roomID) {
        const blockStatusResponse = await this.adminClient.getBlockStatus(roomID);
        if ((0, typescript_result_1.isError)(blockStatusResponse)) {
            return blockStatusResponse;
        }
        else {
            return (0, typescript_result_1.Ok)(blockStatusResponse.ok.block);
        }
    }
    async takedownRoom(roomID) {
        const detailsResponse = await this.roomDetailsProvider.getRoomDetails(roomID);
        let details;
        if ((0, typescript_result_1.isError)(detailsResponse)) {
            log.warn("Unable to fetch details for a room being requested to shutdown", detailsResponse.error);
            details = { room_id: roomID };
        }
        else {
            details = detailsResponse.ok;
        }
        log.debug("Taking down room", roomID);
        // we use delte V1 because clients do not pick up the user's own leave event
        // in V2 and i don't know why.
        // That is very important in the case of stuck invitations.
        const takedownResult = await this.adminClient.shutdownRoomV2(roomID, {
            block: true,
            purge: true,
        });
        if ((0, typescript_result_1.isError)(takedownResult)) {
            return takedownResult;
        }
        else {
            return (0, typescript_result_1.Ok)(details);
        }
    }
    async getRoomDetails(roomID) {
        return await this.roomDetailsProvider.getRoomDetails(roomID);
    }
}
exports.SynapseAdminRoomTakedownCapability = SynapseAdminRoomTakedownCapability;
(0, matrix_protection_suite_1.describeCapabilityProvider)({
    name: SynapseAdminRoomTakedownCapability.name,
    interface: "RoomTakedownCapability",
    description: "Takesdown rooms using the synapse admin API",
    factory(description, draupnir) {
        if (draupnir.synapseAdminClient === undefined) {
            throw new TypeError("Synapse admin client is not available on this draupnir instance");
        }
        return new SynapseAdminRoomTakedownCapability(draupnir.synapseAdminClient);
    },
});
class SimulatedRoomTakedownCapability {
    constructor(adminClient) {
        this.adminClient = adminClient;
        this.requiredPermissions = [];
        this.requiredStatePermissions = [];
        this.requiredEventPermissions = [];
        this.roomDetailsProvider = new SynapseAdminRoomDetailsProvider(this.adminClient);
        // nothing to do mare.
    }
    async isRoomTakendown(roomID) {
        const blockStatusResponse = await this.adminClient.getBlockStatus(roomID);
        if ((0, typescript_result_1.isError)(blockStatusResponse)) {
            return blockStatusResponse;
        }
        else {
            return (0, typescript_result_1.Ok)(blockStatusResponse.ok.block);
        }
    }
    async takedownRoom(roomID) {
        const detailsResponse = await this.roomDetailsProvider.getRoomDetails(roomID);
        let details;
        if ((0, typescript_result_1.isError)(detailsResponse)) {
            log.warn("Unable to fetch details for a room being requested to shutdown", detailsResponse.error);
            details = { room_id: roomID };
        }
        else {
            details = detailsResponse.ok;
        }
        return (0, typescript_result_1.Ok)(details);
    }
    async getRoomDetails(roomID) {
        return await this.roomDetailsProvider.getRoomDetails(roomID);
    }
}
exports.SimulatedRoomTakedownCapability = SimulatedRoomTakedownCapability;
(0, matrix_protection_suite_1.describeCapabilityProvider)({
    name: "SimulatedRoomTakedownCapability",
    description: "Simulates the synapse admin room takedown capability",
    interface: "RoomTakedownCapability",
    factory(description, draupnir) {
        if (draupnir.synapseAdminClient === undefined) {
            throw new TypeError("Synapse admin client is not available on this draupnir instance");
        }
        return new SimulatedRoomTakedownCapability(draupnir.synapseAdminClient);
    },
});
//# sourceMappingURL=SynapseAdminRoomTakedown.js.map