"use strict";
// Copyright 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.MARKDOWN_RENDERER = exports.MarkdownVariables = void 0;
exports.staticString = staticString;
exports.blank = blank;
exports.incrementDynamicEnvironment = incrementDynamicEnvironment;
const DeadDocument_1 = require("./DeadDocument");
var MarkdownVariables;
(function (MarkdownVariables) {
    MarkdownVariables["IndentationLevel"] = "indentation level";
    MarkdownVariables["ListType"] = "List Type";
    MarkdownVariables["ListItemCount"] = "List Item Count";
})(MarkdownVariables || (exports.MarkdownVariables = MarkdownVariables = {}));
function staticString(string) {
    return function (_fringe, _node, context) {
        context.output.writeString(string);
    };
}
function blank() { }
function incrementDynamicEnvironment(_fringe, node, _context, environment) {
    const value = (() => {
        try {
            return environment.read(MarkdownVariables.IndentationLevel);
        }
        catch (_e) {
            return environment.bind(MarkdownVariables.IndentationLevel, node, 0);
        }
    })();
    if (value) {
        if (!Number.isInteger(value)) {
            throw new TypeError(`${MarkdownVariables.IndentationLevel} should not have a dynamic environment entry that isn't an integer`);
        }
        environment.bind(MarkdownVariables.IndentationLevel, node, value + 1);
    }
}
exports.MARKDOWN_RENDERER = new DeadDocument_1.SimpleFringeRenderer();
exports.MARKDOWN_RENDERER.registerRenderer(DeadDocument_1.FringeType.Leaf, DeadDocument_1.NodeTag.TextNode, function (tag, node, context) {
    context.output.writeString(node.data);
})
    .registerInnerNode(DeadDocument_1.NodeTag.HeadingOne, function (_fringeType, _node, context) {
    context.output.writeString("# ");
}, staticString("\n\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.HeadingTwo, function (_fringeType, _node, context) {
    context.output.writeString("## ");
}, staticString("\n\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.HeadingThree, function (_fringeType, _node, context) {
    context.output.writeString("### ");
}, staticString("\n\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.HeadingFour, function (_fringeType, _node, context) {
    context.output.writeString("#### ");
}, staticString("\n\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.HeadingFive, function (_fringeType, _node, context) {
    context.output.writeString("##### ");
}, staticString("\n\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.HeadingSix, function (_fringeType, _node, context) {
    context.output.writeString("###### ");
}, staticString("\n\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.HorizontalRule, blank, staticString("\n---\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.Emphasis, staticString("*"), staticString("*"))
    .registerInnerNode(DeadDocument_1.NodeTag.InlineCode, staticString("`"), staticString("`"))
    .registerInnerNode(DeadDocument_1.NodeTag.Paragraph, blank, staticString("\n\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.PreformattedText, staticString("```\n"), staticString("```\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.Strong, staticString("**"), staticString("**"))
    .registerInnerNode(DeadDocument_1.NodeTag.UnorderedList, function (fringe, node, context, environment) {
    incrementDynamicEnvironment(fringe, node, context, environment);
    environment.bind(MarkdownVariables.ListType, node, DeadDocument_1.NodeTag.UnorderedList);
    environment.bind(MarkdownVariables.ListItemCount, node, 0);
}, blank)
    .registerInnerNode(DeadDocument_1.NodeTag.ListItem, function (_fringe, node, context, environment) {
    const indentationLevel = (() => {
        const value = environment.read(MarkdownVariables.IndentationLevel);
        if (!Number.isInteger(value)) {
            throw new TypeError(`Cannot render the list ${node.tag} because someone clobbered the dynamic environment, should only have integers. Did you forget to enclose in <ul> or <ol>?`);
        }
        else {
            return value;
        }
    })();
    const listItemCount = (() => {
        const currentCount = environment.read(MarkdownVariables.ListItemCount);
        if (!Number.isInteger(currentCount)) {
            throw new TypeError(`Cannot render the list ${node.tag} because someone clobbered the dynamic environment.`);
        }
        environment.write(MarkdownVariables.ListItemCount, currentCount + 1);
        return currentCount + 1;
    })();
    context.output.writeString("\n");
    for (let i = 0; i < indentationLevel; i++) {
        context.output.writeString("    ");
    }
    if (environment.read(MarkdownVariables.ListType) === DeadDocument_1.NodeTag.OrderedList) {
        context.output.writeString(` ${listItemCount}. `);
    }
    else {
        context.output.writeString(" * ");
    }
}, staticString("\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.OrderedList, function (fringe, node, context, environment) {
    incrementDynamicEnvironment(fringe, node, context, environment);
    environment.bind(MarkdownVariables.ListType, node, DeadDocument_1.NodeTag.OrderedList);
    environment.bind(MarkdownVariables.ListItemCount, node, 0);
}, blank)
    .registerInnerNode(DeadDocument_1.NodeTag.LineBreak, blank, staticString("\n"))
    .registerInnerNode(DeadDocument_1.NodeTag.BoldFace, staticString("**"), staticString("**"))
    .registerInnerNode(DeadDocument_1.NodeTag.ItalicFace, staticString("*"), staticString("*"))
    .registerInnerNode(DeadDocument_1.NodeTag.Anchor, staticString("["), function (_fringe, node, context, _environment) {
    const href = node.attributeMap.get("href");
    if (href === undefined) {
        throw new TypeError("Anchor without a href is probably a mistake? well we do not support other uses yet.");
    }
    context.output.writeString(`](${href})`);
})
    .registerInnerNode(DeadDocument_1.NodeTag.Root, blank, blank)
    .registerInnerNode(DeadDocument_1.NodeTag.Details, staticString("<details>"), staticString("</details>"))
    .registerInnerNode(DeadDocument_1.NodeTag.Summary, staticString("<summary>"), staticString("</summary>"))
    .registerInnerNode(DeadDocument_1.NodeTag.Font, blank, blank)
    .registerInnerNode(DeadDocument_1.NodeTag.Span, blank, blank);
//# sourceMappingURL=DeadDocumentMarkdown.js.map