"use strict";
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from Draupnir
// https://github.com/the-draupnir-project/Draupnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderParameterDescription = renderParameterDescription;
exports.renderCommandSummary = renderCommandSummary;
exports.renderCommandHelp = renderCommandHelp;
exports.renderErrorDetails = renderErrorDetails;
exports.replyToEventWithErrorDetails = replyToEventWithErrorDetails;
exports.renderActionResultToEvent = renderActionResultToEvent;
exports.reactToEventWithResult = reactToEventWithResult;
exports.matrixCommandRenderer = matrixCommandRenderer;
exports.renderMentionPill = renderMentionPill;
exports.renderRoomPill = renderRoomPill;
exports.renderTableHelp = renderTableHelp;
exports.wrapInRoot = wrapInRoot;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const MPSMatrixInterfaceAdaptor_1 = require("./MPSMatrixInterfaceAdaptor");
const PresentationSchema_1 = require("@the-draupnir-project/interface-manager/dist/Command/PresentationSchema");
const replyNotice_1 = require("./replyNotice");
const CommonRenderers_1 = require("./CommonRenderers");
const log = new matrix_protection_suite_1.Logger("MatrixHelpRenderer");
function requiredArgument(argumentName) {
    return `<${argumentName}>`;
}
function keywordArgument(keyword) {
    // ahh fuck what about defaults for keys?
    return `[--${keyword}]`;
}
// they should be allowed to name the rest argument...
function restArgument(rest) {
    return `[...${rest.name}]`;
}
function renderParameterDescription(description) {
    var _a;
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        description.name,
        " - ", (_a = description.description) !== null && _a !== void 0 ? _a : "no description",
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null)));
}
function renderCommandSummary(command, tableEntry) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, renderCommandHelp(command, tableEntry.designator)),
            " -",
            " ",
            command.summary),
        command.description ? (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Description:"),
            interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
            command.description,
            interface_manager_1.DeadDocumentJSX.JSXFactory("br", null))) : (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null)),
        command.parametersDescription.descriptions.length > 0 ? (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Parameters:"),
            interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
            ...command.parametersDescription.descriptions.map(renderParameterDescription))) : (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null))));
}
function renderCommandHelp(command, designator) {
    const rest = command.parametersDescription.rest;
    const keywords = command.parametersDescription.keywords;
    return [
        ...designator,
        ...command.parametersDescription.descriptions.map((d) => requiredArgument(d.name)),
        ...(rest ? [restArgument(rest)] : []),
        ...Object.keys(keywords.keywordDescriptions).map((k) => keywordArgument(k)),
    ].join(" ");
}
function renderErrorDetails(error) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null, error.mostRelevantElaboration),
        (0, CommonRenderers_1.renderDetailsNotice)(error),
        (0, CommonRenderers_1.renderElaborationTrail)(error),
        (0, CommonRenderers_1.renderExceptionTrail)(error)));
}
async function replyToEventWithErrorDetails(roomMessageSender, event, error) {
    return (await (0, MPSMatrixInterfaceAdaptor_1.sendMatrixEventsFromDeadDocument)(roomMessageSender, event.room_id, interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, renderErrorDetails(error)), { replyToEvent: event }));
}
function renderActionResultToEvent(roomMessageSender, roomReactionSender, event, result) {
    if ((0, matrix_protection_suite_1.isError)(result)) {
        void (0, matrix_protection_suite_1.Task)(replyToEventWithErrorDetails(roomMessageSender, event, result.error));
    }
    void (0, matrix_protection_suite_1.Task)(reactToEventWithResult(roomReactionSender, event, result));
}
// Maybe we need something like the MatrixInterfaceAdaptor but for Error types?
function formattedArgumentHint(error) {
    const argumentsUpToError = error.partialCommand.stream.source.slice(0, error.partialCommand.stream.getPosition());
    let commandContext = "Command context:";
    for (const argument of argumentsUpToError) {
        commandContext += ` ${interface_manager_1.TextPresentationRenderer.render(argument)}`;
    }
    const badArgument = error.partialCommand.stream.peekItem();
    const badArgumentHint = ` ${badArgument === undefined
        ? "undefined"
        : interface_manager_1.TextPresentationRenderer.render(badArgument)}\n${Array(commandContext.length + 1).join(" ")} ^ expected ${(0, PresentationSchema_1.printPresentationSchema)(error.parameter.acceptor)} here`;
    return commandContext + badArgumentHint;
}
async function reactToEventWithResult(roomReactionSender, event, result) {
    // implement this so we can use it in the invitation protection
    // then in the invitation protection makes ure we render when the listener fails
    // then in the ban propagation protection also do this.
    const react = async (emote) => {
        return (await roomReactionSender.sendReaction(event.room_id, event.event_id, emote));
    };
    if ((0, matrix_protection_suite_1.isOk)(result)) {
        return await react("✅");
    }
    else {
        return await react("❌");
    }
}
function renderArgumentParseError(error) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
        "There was a problem when parsing the ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, error.parameter.name),
        " ",
        "parameter for this command.",
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        renderCommandHelp(error.partialCommand.description, error.partialCommand.designator),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        error.message,
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        interface_manager_1.DeadDocumentJSX.JSXFactory("pre", null, formattedArgumentHint(error))));
}
function renderUnexpectedArgumentError(error) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
        "There was an unexpected argument provided for this command.",
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        renderCommandHelp(error.partialCommand.description, error.partialCommand.designator),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        error.message,
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null)));
}
async function matrixCommandRenderer({ clientPlatform, commandRoomID }, { event }, _command, result) {
    void (0, matrix_protection_suite_1.Task)(reactToEventWithResult(clientPlatform.toRoomReactionSender(), event, result));
    if ((0, matrix_protection_suite_1.isError)(result)) {
        if (result.error instanceof interface_manager_1.ArgumentParseError) {
            return (await (0, MPSMatrixInterfaceAdaptor_1.sendMatrixEventsFromDeadDocument)(clientPlatform.toRoomMessageSender(), commandRoomID, renderArgumentParseError(result.error), { replyToEvent: event }));
        }
        else if (result.error instanceof interface_manager_1.UnexpectedArgumentError) {
            return (await (0, MPSMatrixInterfaceAdaptor_1.sendMatrixEventsFromDeadDocument)(clientPlatform.toRoomMessageSender(), commandRoomID, renderUnexpectedArgumentError(result.error), { replyToEvent: event }));
        }
        else if (result.error instanceof matrix_protection_suite_1.ActionException) {
            const commandError = result.error;
            log.error("command error", commandError.uuid, commandError.message, commandError.exception);
            return (await (0, MPSMatrixInterfaceAdaptor_1.sendMatrixEventsFromDeadDocument)(clientPlatform.toRoomMessageSender(), commandRoomID, renderCommandException(result.error), { replyToEvent: event }));
        }
        else {
            const noticeResult = await (0, replyNotice_1.replyNoticeText)(clientPlatform.toRoomMessageSender(), commandRoomID, event.event_id, `An unexpected error occurred while processing the command: ${result.error.message}`);
            if ((0, matrix_protection_suite_1.isError)(noticeResult)) {
                return noticeResult.elaborate("Could not reply to a command to report an error back to the user");
            }
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
    }
    return (0, matrix_protection_suite_1.Ok)(undefined);
}
function renderCommandException(error) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
        "There was an unexpected error when processing this command:",
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        error.message,
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        "Details can be found by providing the reference ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, error.uuid),
        "to an administrator."));
}
function renderMentionPill(mxid, displayName) {
    const url = `https://matrix.to/#/${mxid}`;
    return interface_manager_1.DeadDocumentJSX.JSXFactory("a", { href: url }, displayName);
}
function renderRoomPill(room) {
    return interface_manager_1.DeadDocumentJSX.JSXFactory("a", { href: room.toPermalink() }, room.toRoomIDOrAlias());
}
function sortCommandsBySourceTable(table) {
    var _a;
    const commandsBySourceTable = new Map();
    for (const command of table.getAllCommands()) {
        const sourceTable = command.sourceTable;
        const groupedCommands = (_a = commandsBySourceTable.get(sourceTable)) !== null && _a !== void 0 ? _a : ((groupedCommands) => (commandsBySourceTable.set(sourceTable, groupedCommands), groupedCommands))([]);
        groupedCommands.push(command);
    }
    return commandsBySourceTable;
}
function sortGroupedCommandsByDesignator(commandsBySourceTable) {
    for (const commands of commandsBySourceTable.values()) {
        commands.sort((a, b) => {
            const aDesignator = a.designator.join(".");
            const bDesignator = b.designator.join(".");
            return aDesignator.localeCompare(bDesignator);
        });
    }
    return commandsBySourceTable;
}
function renderSourceTableSummary(sourceTable, sortedEntries) {
    const tableName = typeof sourceTable.name === "string"
        ? sourceTable.name.charAt(0).toUpperCase() + sourceTable.name.slice(1)
        : sourceTable.name.toString();
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("b", null,
                    tableName,
                    " commands:")),
            sortedEntries.map((entry) => renderCommandSummary(entry.currentCommand, entry)))));
}
function renderTableHelp(table, documentationURL) {
    const groupedAndSortedCommands = sortGroupedCommandsByDesignator(sortCommandsBySourceTable(table));
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Documentation: "),
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("a", { href: documentationURL }, documentationURL),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        [...groupedAndSortedCommands.entries()].map(([sourceTable, sortedEntries]) => renderSourceTableSummary(sourceTable, sortedEntries))));
}
function wrapInRoot(node) {
    return interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, node);
}
//# sourceMappingURL=MatrixHelpRenderer.js.map