"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BotSDKBaseClient = void 0;
exports.is404 = is404;
exports.resultifyBotSDKRequestErrorWith404AsUndefined = resultifyBotSDKRequestErrorWith404AsUndefined;
exports.resultifyBotSDKRequestError = resultifyBotSDKRequestError;
const typebox_1 = require("@sinclair/typebox");
const matrix_bot_sdk_1 = require("matrix-bot-sdk");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const SafeMatrixClient_1 = require("../SafeMatrixClient");
const util_1 = __importDefault(require("util"));
const log = new matrix_protection_suite_1.Logger('BotSDKBaseClient');
const WeakError = typebox_1.Type.Object({
    message: typebox_1.Type.String(),
    name: typebox_1.Type.String(),
});
function toRoomID(room) {
    return typeof room === 'string' ? room : room.toRoomIDOrAlias();
}
function matrixExceptionFromMatrixError(error) {
    return matrix_protection_suite_1.MatrixException.R({
        exception: error,
        matrixErrorCode: error.errcode,
        matrixErrorMessage: error.error,
        message: error.message,
    });
}
function actionExceptionFromWeakError(error) {
    return matrix_protection_suite_1.ActionException.Result(error.message, {
        exception: error,
        exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown,
    });
}
function unknownError(error) {
    const printedError = (() => {
        if (typeof error === 'object' && error !== null) {
            // eslint-disable-next-line @typescript-eslint/no-base-to-string
            const toString = error.toString();
            if (toString !== '[object Object]') {
                return toString;
            }
        }
        try {
            return JSON.stringify(error);
        }
        catch (_a) {
            return util_1.default.inspect(error, {
                depth: 2,
                maxArrayLength: 10,
                breakLength: 80,
            });
        }
    })();
    throw new TypeError(`What on earth are you throwing exactly? because it isn't an error: ${printedError}`);
}
function is404(error) {
    return (typeof error === 'object' &&
        error !== null &&
        'statusCode' in error &&
        error.statusCode === 404);
}
// Either i'm really tired right now or stupid.
// But I can't think of a way to share this definition with
// `resultifyBotSDKRequestError` without having never | undefined
// clients need to just have `never` when 404 isn't being checked!
function resultifyBotSDKRequestErrorWith404AsUndefined(error) {
    if (is404(error)) {
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
    if (error instanceof matrix_bot_sdk_1.MatrixError) {
        return matrixExceptionFromMatrixError(error);
    }
    else if (matrix_protection_suite_1.Value.Check(WeakError, error)) {
        return actionExceptionFromWeakError(error);
    }
    else {
        unknownError(error);
    }
}
function resultifyBotSDKRequestError(error) {
    if (error instanceof matrix_bot_sdk_1.MatrixError) {
        return matrixExceptionFromMatrixError(error);
    }
    else if (matrix_protection_suite_1.Value.Check(WeakError, error)) {
        return actionExceptionFromWeakError(error);
    }
    else {
        unknownError(error);
    }
}
class BotSDKBaseClient {
    constructor(client, clientUserID, clientRooms, eventDecoder) {
        this.client = client;
        this.clientUserID = clientUserID;
        this.clientRooms = clientRooms;
        this.eventDecoder = eventDecoder;
        // nothing to do.
    }
    preemptTimelineJoin(_roomID) {
        // nothing to do.
    }
    async getClientCapabilities() {
        return await this.client
            .doRequest('GET', '/_matrix/client/v3/capabilities')
            .then((value) => matrix_protection_suite_1.Value.Decode(matrix_protection_suite_1.ClientCapabilitiesResponse, value), resultifyBotSDKRequestError);
    }
    async sendMessage(roomID, content) {
        return await this.client
            .sendMessage(roomID, content)
            .then((eventID) => (0, matrix_protection_suite_1.Ok)(eventID), resultifyBotSDKRequestError);
    }
    async getEvent(roomID, eventID) {
        return await this.client
            .getEvent(roomID, eventID)
            .then((event) => this.eventDecoder.decodeEvent(event), resultifyBotSDKRequestError);
    }
    async getUndecodedEvent(roomID, eventID) {
        return await this.client
            .getEvent(roomID, eventID)
            .then((event) => (0, matrix_protection_suite_1.Ok)(event), resultifyBotSDKRequestError);
    }
    async sendReaction(roomID, eventID, key) {
        return await this.client.unstableApis
            .addReactionToEvent(roomID, eventID, key)
            .then((eventID) => (0, matrix_protection_suite_1.Ok)(eventID), resultifyBotSDKRequestError);
    }
    async resolveRoom(room) {
        const roomReference = (() => {
            if (typeof room === 'string') {
                return matrix_basic_types_1.MatrixRoomReference.fromRoomIDOrAlias(room);
            }
            else {
                return room;
            }
        })();
        return await (0, SafeMatrixClient_1.resolveRoomReferenceSafe)(this.client, roomReference);
    }
    async inviteUser(room, userID, reason) {
        const roomID = room instanceof matrix_basic_types_1.MatrixRoomID ? room.toRoomIDOrAlias() : room;
        return await this.client
            .doRequest('POST', `/_matrix/client/v3/rooms/${encodeURIComponent(roomID)}/invite`, null, {
            user_id: userID,
            ...(reason ? { reason } : {}),
        })
            .then((_) => (0, matrix_protection_suite_1.Ok)(undefined), resultifyBotSDKRequestError);
    }
    async joinRoom(room, rawOptions) {
        var _a;
        const alwaysCallJoin = (_a = rawOptions === null || rawOptions === void 0 ? void 0 : rawOptions.alwaysCallJoin) !== null && _a !== void 0 ? _a : false;
        const resolvedReference = await this.resolveRoom(room);
        if ((0, matrix_protection_suite_1.isError)(resolvedReference)) {
            return resolvedReference;
        }
        if (!alwaysCallJoin &&
            this.clientRooms.isJoinedRoom(resolvedReference.ok.toRoomIDOrAlias())) {
            return resolvedReference;
        }
        return await this.client
            .joinRoom(resolvedReference.ok.toRoomIDOrAlias(), resolvedReference.ok.getViaServers())
            .then((roomID) => {
            this.preemptTimelineJoin(roomID);
            return (0, matrix_protection_suite_1.Ok)(matrix_basic_types_1.MatrixRoomReference.fromRoomID(roomID, resolvedReference.ok.getViaServers()));
        }, resultifyBotSDKRequestError);
    }
    async createRoom(options) {
        return await this.client.createRoom(options).then((roomID) => {
            this.preemptTimelineJoin(roomID);
            return (0, matrix_protection_suite_1.Ok)(matrix_basic_types_1.MatrixRoomReference.fromRoomID(roomID, [
                (0, matrix_basic_types_1.userServerName)(this.clientUserID),
            ]));
        }, resultifyBotSDKRequestError);
    }
    async banUser(room, userID, reason) {
        return await this.client
            .banUser(userID, toRoomID(room), reason)
            .then((_) => (0, matrix_protection_suite_1.Ok)(undefined), resultifyBotSDKRequestError);
    }
    async kickUser(room, userID, reason) {
        return await this.client
            .kickUser(userID, toRoomID(room), reason)
            .then((_) => (0, matrix_protection_suite_1.Ok)(undefined), resultifyBotSDKRequestError);
    }
    async redactEvent(room, eventID, reason) {
        return await this.client
            .redactEvent(toRoomID(room), eventID, reason)
            .then((redactionEventID) => (0, matrix_protection_suite_1.Ok)(redactionEventID), resultifyBotSDKRequestError);
    }
    toRoomEventRelationsPaginator(roomID, eventID) {
        return Object.freeze({
            client: this.client,
            eventDecoder: this.eventDecoder,
            async fetchPage(ergonomicOptions) {
                const options = {
                    ...ergonomicOptions,
                    dir: ergonomicOptions.direction === 'forwards' ? 'f' : 'b',
                };
                return this.client
                    .doRequest('GET', `/_matrix/client/v1/rooms/${encodeURIComponent(roomID)}/relations/${encodeURIComponent(eventID)}`, options)
                    .then((response) => {
                    const firstSchemaPass = matrix_protection_suite_1.Value.Decode(matrix_protection_suite_1.RoomEventRelationsResponse, response);
                    if ((0, matrix_protection_suite_1.isError)(firstSchemaPass)) {
                        return firstSchemaPass.elaborate('The response for /relations is severly malformed');
                    }
                    const parsedEvents = [];
                    for (const event of firstSchemaPass.ok.chunk) {
                        const decodedEvent = this.eventDecoder.decodeEvent(event);
                        if ((0, matrix_protection_suite_1.isError)(decodedEvent)) {
                            return decodedEvent.elaborate('Failed to decode an event in the /relations response');
                        }
                        parsedEvents.push(decodedEvent.ok);
                    }
                    return (0, matrix_protection_suite_1.Ok)({
                        previousToken: firstSchemaPass.ok.prev_batch,
                        nextToken: firstSchemaPass.ok.next_batch,
                        chunk: parsedEvents,
                        hasNext: firstSchemaPass.ok.next_batch !== undefined,
                        hasPrevious: true,
                    });
                });
            },
        });
    }
    toRoomEventRelationsIterator(roomID, eventID, options) {
        return new matrix_protection_suite_1.StandardPaginationIterator(options, this.toRoomEventRelationsPaginator(roomID, eventID));
    }
    async getAllState(room) {
        const decodeResults = await this.client
            .getRoomState(toRoomID(room))
            .then((events) => (0, matrix_protection_suite_1.Ok)(events.map((event) => this.eventDecoder.decodeStateEvent(event))), resultifyBotSDKRequestError);
        if ((0, matrix_protection_suite_1.isError)(decodeResults)) {
            return decodeResults;
        }
        const errors = [];
        const events = [];
        for (const result of decodeResults.ok) {
            if ((0, matrix_protection_suite_1.isError)(result)) {
                errors.push(result.error);
            }
            else {
                events.push(result.ok);
            }
        }
        if (errors.length > 0) {
            log.error(`There were multiple errors while decoding state events for ${room.toString()}`, matrix_protection_suite_1.MultipleErrors.Result(`Unable to decode state events in ${room.toString()}`, { errors }));
        }
        return (0, matrix_protection_suite_1.Ok)(events);
    }
    async unbanUser(room, userID) {
        return await this.client
            .unbanUser(userID, toRoomID(room))
            .then((_) => (0, matrix_protection_suite_1.Ok)(undefined), resultifyBotSDKRequestError);
    }
    async sendStateEvent(room, stateType, stateKey, content) {
        const roomID = room instanceof matrix_basic_types_1.MatrixRoomID ? room.toRoomIDOrAlias() : room;
        return await this.client
            .sendStateEvent(roomID, stateType, stateKey, content)
            .then((eventID) => (0, matrix_protection_suite_1.Ok)(eventID), resultifyBotSDKRequestError);
    }
    toRoomMessagesPaginator(roomID) {
        return Object.freeze({
            client: this.client,
            eventDecoder: this.eventDecoder,
            async fetchPage(ergonomicOptions) {
                const options = {
                    ...ergonomicOptions,
                    ...(ergonomicOptions.filter
                        ? { filter: JSON.stringify(ergonomicOptions.filter) }
                        : {}),
                    dir: ergonomicOptions.direction === 'forwards' ? 'f' : 'b',
                };
                return this.client
                    .doRequest('GET', `/_matrix/client/v3/rooms/${encodeURIComponent(roomID)}/messages`, options)
                    .then((response) => {
                    const firstSchemaPass = matrix_protection_suite_1.Value.Decode(matrix_protection_suite_1.RoomMessagesResponse, response);
                    if ((0, matrix_protection_suite_1.isError)(firstSchemaPass)) {
                        return firstSchemaPass.elaborate('The response for /messages is severly malformed');
                    }
                    const parsedEvents = [];
                    for (const event of firstSchemaPass.ok.chunk) {
                        const decodedEvent = this.eventDecoder.decodeEvent(event);
                        if ((0, matrix_protection_suite_1.isError)(decodedEvent)) {
                            return decodedEvent.elaborate('Failed to decode an event in the /messages response');
                        }
                        parsedEvents.push(decodedEvent.ok);
                    }
                    return (0, matrix_protection_suite_1.Ok)({
                        previousToken: firstSchemaPass.ok.start,
                        nextToken: firstSchemaPass.ok.end,
                        chunk: parsedEvents,
                        hasNext: firstSchemaPass.ok.end !== undefined,
                        hasPrevious: true,
                    });
                });
            },
        });
    }
    toRoomMessagesIterator(roomID, options) {
        return new matrix_protection_suite_1.StandardPaginationIterator(options, this.toRoomMessagesPaginator(roomID));
    }
}
exports.BotSDKBaseClient = BotSDKBaseClient;
//# sourceMappingURL=BotSDKBaseClient.js.map