"use strict";
/**
 * Copyright (C) 2023 Gnuxie <Gnuxie@protonmail.com>
 * All rights reserved.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.RoomStateManagerFactory = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const RoomMembershipManager_1 = require("../StateTracking/RoomMembershipManager");
const PolicyListManager_1 = require("../PolicyList/PolicyListManager");
const BotSDKClientPlatform_1 = require("../Client/BotSDKClientPlatform");
const BotSDKBaseClient_1 = require("../Client/BotSDKBaseClient");
const RoomStateRefresh_1 = require("./RoomStateRefresh");
const log = new matrix_protection_suite_1.Logger('RoomStateManagerFactory');
class RoomStateManagerFactory {
    constructor(clientsInRoomMap, clientProvider, eventDecoder, roomStateBackingStore, hashStore) {
        this.clientsInRoomMap = clientsInRoomMap;
        this.clientProvider = clientProvider;
        this.eventDecoder = eventDecoder;
        this.roomStateBackingStore = roomStateBackingStore;
        this.hashStore = hashStore;
        this.roomStateRefresher = new RoomStateRefresh_1.RoomStateRefresh();
        this.roomStateIssuers = new matrix_protection_suite_1.InternedInstanceFactory(async (_roomID, room) => {
            const roomStateGetterResult = await this.getRoomStateGetterForRevisionIssuer(room);
            if ((0, matrix_protection_suite_1.isError)(roomStateGetterResult)) {
                return roomStateGetterResult;
            }
            const getInitialRoomState = async () => {
                if (this.roomStateBackingStore !== undefined) {
                    const storeResult = await this.roomStateBackingStore.getRoomState(room.toRoomIDOrAlias());
                    if ((0, matrix_protection_suite_1.isOk)(storeResult)) {
                        if (storeResult.ok !== undefined) {
                            return (0, matrix_protection_suite_1.Ok)({ state: storeResult.ok, isFromStore: true });
                        }
                    }
                    else {
                        log.error(`Could not load room state from the backing store`, storeResult.error);
                    }
                }
                const stateRequestResult = await roomStateGetterResult.ok.getAllState(room);
                if ((0, matrix_protection_suite_1.isError)(stateRequestResult)) {
                    return stateRequestResult;
                }
                return (0, matrix_protection_suite_1.Ok)({ state: stateRequestResult.ok, isFromStore: false });
            };
            const stateResult = await getInitialRoomState();
            // TODO: This entire class needs moving the MPS main via client capabilities.
            //       so that it can be unit tested.
            if ((0, matrix_protection_suite_1.isError)(stateResult)) {
                return stateResult;
            }
            const issuer = new matrix_protection_suite_1.StandardRoomStateRevisionIssuer(room, roomStateGetterResult.ok, stateResult.ok.state);
            if (this.roomStateBackingStore) {
                issuer.on('revision', this.roomStateBackingStore.revisionListener);
            }
            // Refresh the state if it was loaded from the store
            if (stateResult.ok.isFromStore) {
                this.roomStateRefresher.refreshState(issuer);
            }
            return (0, matrix_protection_suite_1.Ok)(issuer);
        });
        this.policyRoomIssuers = new matrix_protection_suite_1.InternedInstanceFactory(async (_key, room) => {
            var _a;
            const roomStateIssuer = await this.roomStateIssuers.getInstance(room.toRoomIDOrAlias(), room);
            if ((0, matrix_protection_suite_1.isError)(roomStateIssuer)) {
                return roomStateIssuer;
            }
            const issuer = new matrix_protection_suite_1.RoomStatePolicyRoomRevisionIssuer(room, matrix_protection_suite_1.StandardPolicyRoomRevision.blankRevision(room), roomStateIssuer.ok);
            (_a = this.sha256Reverser) === null || _a === void 0 ? void 0 : _a.addPolicyRoomRevisionIssuer(issuer);
            return (0, matrix_protection_suite_1.Ok)(issuer);
        });
        this.roomMembershipIssuers = new matrix_protection_suite_1.InternedInstanceFactory(async (_roomID, room) => {
            const stateIssuer = await this.roomStateIssuers.getInstance(room.toRoomIDOrAlias(), room);
            if ((0, matrix_protection_suite_1.isError)(stateIssuer)) {
                return stateIssuer;
            }
            return (0, matrix_protection_suite_1.Ok)(new matrix_protection_suite_1.RoomStateMembershipRevisionIssuer(room, matrix_protection_suite_1.StandardRoomMembershipRevision.blankRevision(room).reviseFromMembership(stateIssuer.ok.currentRevision.getStateEventsOfType('m.room.member')), stateIssuer.ok));
        });
        this.sha256Reverser = this.hashStore
            ? new matrix_protection_suite_1.StandardSHA256HashReverser(this.hashStore)
            : undefined;
        // nothing to do.
    }
    async getRoomStateGetterForRevisionIssuer(room) {
        const managedClientsInRoom = this.clientsInRoomMap.getManagedUsersInRoom(room.toRoomIDOrAlias());
        const chosenClientUserID = managedClientsInRoom[0];
        if (chosenClientUserID === undefined) {
            return matrix_protection_suite_1.ActionError.Result(`There is no managed client in the room ${room.toPermalink()} and so we cannot fetch the room state there.`);
        }
        const client = await this.clientProvider(chosenClientUserID);
        const clientRooms = this.clientsInRoomMap.getClientRooms(chosenClientUserID);
        if (clientRooms === undefined) {
            throw new TypeError(`Cannot find clientRooms for ${chosenClientUserID}`);
        }
        return (0, matrix_protection_suite_1.Ok)(new BotSDKClientPlatform_1.BotSDKClientPlatform(new BotSDKBaseClient_1.BotSDKBaseClient(client, chosenClientUserID, clientRooms, this.eventDecoder)).toRoomStateGetter());
    }
    requestingUserNotJoined(clientUserID, room) {
        const message = `The user ${clientUserID} is not joined to the room ${room.toPermalink()}`;
        return new matrix_protection_suite_1.ActionException(matrix_protection_suite_1.ActionExceptionKind.Unknown, new Error(message), message);
    }
    async getRoomStateRevisionIssuer(room, clientUserID) {
        const roomID = room.toRoomIDOrAlias();
        if (this.clientsInRoomMap.isClientPreemptivelyInRoom(clientUserID, roomID)) {
            return await this.roomStateIssuers.getInstance(room.toRoomIDOrAlias(), room);
        }
        else {
            return (0, matrix_protection_suite_1.ResultError)(this.requestingUserNotJoined(clientUserID, room));
        }
    }
    async getRoomStateManager(clientUserID) {
        return new BotSDKRoomStateManager(clientUserID, this);
    }
    async getPolicyRoomRevisionIssuer(room, clientUserID) {
        const roomID = room.toRoomIDOrAlias();
        if (this.clientsInRoomMap.isClientPreemptivelyInRoom(clientUserID, roomID)) {
            return await this.policyRoomIssuers.getInstance(roomID, room);
        }
        else {
            return (0, matrix_protection_suite_1.ResultError)(this.requestingUserNotJoined(clientUserID, room));
        }
    }
    getEditablePolicyRoomIDs(editor, ruleType) {
        const editableRoomIDs = this.policyRoomIssuers
            .allInstances()
            .filter((issuer) => issuer.currentRevision.isAbleToEdit(editor, ruleType))
            .map((issuer) => issuer.currentRevision.room);
        return editableRoomIDs;
    }
    async getPolicyRoomManager(clientUserID) {
        const client = await this.clientProvider(clientUserID);
        const clientRooms = this.clientsInRoomMap.getClientRooms(clientUserID);
        if (clientRooms === undefined) {
            throw new TypeError(`Cannot find clientRooms for ${clientUserID}`);
        }
        // FIXME: Shouldn't we have an equivalent of the clientProvider that
        // gives us a clientPlatform? or one that gives both the platform and the client?
        return new PolicyListManager_1.BotSDKPolicyRoomManager(clientUserID, client, new BotSDKClientPlatform_1.BotSDKClientPlatform(new BotSDKBaseClient_1.BotSDKBaseClient(client, clientUserID, clientRooms, this.eventDecoder)), this, this.clientsInRoomMap);
    }
    async getRoomMembershipRevisionIssuer(room, clientUserID) {
        const roomID = room.toRoomIDOrAlias();
        if (this.clientsInRoomMap.isClientPreemptivelyInRoom(clientUserID, roomID)) {
            return await this.roomMembershipIssuers.getInstance(roomID, room);
        }
        else {
            return (0, matrix_protection_suite_1.ResultError)(this.requestingUserNotJoined(clientUserID, room));
        }
    }
    async getRoomMembershipManager(clientUserID) {
        const client = await this.clientProvider(clientUserID);
        return new RoomMembershipManager_1.BotSDKRoomMembershipManager(clientUserID, client, this);
    }
    handleTimelineEvent(roomID, event) {
        if (this.roomStateIssuers.hasInstance(roomID) &&
            ('state_key' in event || event.type === 'm.room.redaction')) {
            const issuer = this.roomStateIssuers.getStoredInstance(roomID);
            if (issuer === undefined) {
                throw new TypeError('Somehow the has method for the interned instances is lying or the code is wrong');
            }
            if (event.type === 'm.room.redaction') {
                issuer.updateForRedaction(event);
            }
            else {
                issuer.updateForEvent(event);
            }
        }
    }
}
exports.RoomStateManagerFactory = RoomStateManagerFactory;
class BotSDKRoomStateManager {
    constructor(clientUserID, factory) {
        this.clientUserID = clientUserID;
        this.factory = factory;
        // nothing to do.
    }
    async getRoomStateRevisionIssuer(room) {
        return await this.factory.getRoomStateRevisionIssuer(room, this.clientUserID);
    }
}
//# sourceMappingURL=RoomStateManagerFactory.js.map