"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerACLConequences = void 0;
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const Action_1 = require("../../../Interface/Action");
const AccessControl_1 = require("../../AccessControl");
const CapabilityProvider_1 = require("../CapabilityProvider");
const RoomSetResult_1 = require("./RoomSetResult");
require("./ServerConsequences"); // we need this so the interface is loaded.
const ActionException_1 = require("../../../Interface/ActionException");
class ServerACLQueue {
    constructor(stateEventSender, serverName, protectedRoomsSet) {
        this.stateEventSender = stateEventSender;
        this.serverName = serverName;
        this.protectedRoomsSet = protectedRoomsSet;
        this.pendingRoomChecks = new Map();
        this.activeRoomChecks = new Map();
        // nothing to do.
    }
    async applyPolicyRevisionToRoom(roomID, issuer) {
        const ACL = AccessControl_1.AccessControl.compileServerACL(this.serverName, issuer.currentRevision);
        const stateRevision = this.protectedRoomsSet.setRoomState.getRevision(roomID);
        if (stateRevision === undefined) {
            throw new TypeError(`Somehowe we can't get the state for this room ${roomID}`);
        }
        const existingStateEvent = stateRevision.getStateEvent('m.room.server_acl', '');
        if (existingStateEvent !== undefined &&
            ACL.matches(existingStateEvent.content)) {
            return (0, Action_1.Ok)(false);
        }
        const result = await this.stateEventSender.sendStateEvent(roomID, 'm.room.server_acl', '', ACL.safeAclContent());
        if ((0, Action_1.isError)(result)) {
            return result;
        }
        else {
            return (0, Action_1.Ok)(true);
        }
    }
    async doActiveCheck(roomID, revisionIssuer) {
        try {
            const activeCheck = this.applyPolicyRevisionToRoom(roomID, revisionIssuer);
            this.activeRoomChecks.set(roomID, activeCheck);
            return await activeCheck;
        }
        finally {
            this.activeRoomChecks.delete(roomID);
        }
    }
    async enqueueCheck(roomID, revisionIssuer, activeCheck) {
        try {
            await activeCheck;
        }
        finally {
            this.pendingRoomChecks.delete(roomID);
        }
        return await this.doActiveCheck(roomID, revisionIssuer);
    }
    async enqueueRoomCheck(roomID, revisionIssuer) {
        const pendingCheck = this.pendingRoomChecks.get(roomID);
        if (pendingCheck) {
            return pendingCheck;
        }
        const activeCheck = this.activeRoomChecks.get(roomID);
        if (activeCheck) {
            const pendingCheck = this.enqueueCheck(roomID, revisionIssuer, activeCheck);
            this.pendingRoomChecks.set(roomID, pendingCheck);
            return await pendingCheck;
        }
        else {
            return await this.doActiveCheck(roomID, revisionIssuer);
        }
    }
}
class ServerACLConequences {
    constructor(stateEventSender, protectedRoomsSet) {
        this.protectedRoomsSet = protectedRoomsSet;
        this.requiredPermissions = [];
        this.requiredEventPermissions = [];
        this.requiredStatePermissions = ['m.room.server_acl'];
        this.queue = new ServerACLQueue(stateEventSender, (0, matrix_basic_types_1.userServerName)(this.protectedRoomsSet.userID), protectedRoomsSet);
    }
    async applyPolicyRevisionToSet(revisionIssuer) {
        const resultBuilder = new RoomSetResult_1.RoomSetResultBuilder();
        try {
            await Promise.all(this.protectedRoomsSet.allProtectedRooms.map((room) => this.queue
                .enqueueRoomCheck(room.toRoomIDOrAlias(), revisionIssuer)
                .then((result) => {
                resultBuilder.addResult(room.toRoomIDOrAlias(), result);
            })));
        }
        catch (e) {
            if (e instanceof Error) {
                return ActionException_1.ActionException.Result(`Uncaught error while applying server ACLS`, {
                    exception: e,
                    exceptionKind: ActionException_1.ActionExceptionKind.Unknown,
                });
            }
        }
        return (0, Action_1.Ok)(resultBuilder.getResult());
    }
    async consequenceForServersInRoom(roomID, revisionIssuer) {
        return await this.queue.enqueueRoomCheck(roomID, revisionIssuer);
    }
    async consequenceForServersInRoomSet(revisionIssuer) {
        return await this.applyPolicyRevisionToSet(revisionIssuer);
    }
    async unbanServerFromRoomSet(serverName, _reason) {
        var _a, _b;
        const revisionIssuer = this.protectedRoomsSet.watchedPolicyRooms.revisionIssuer;
        const revision = revisionIssuer.currentRevision;
        // this method is for applying the ACL, not removing policies.
        const access = AccessControl_1.AccessControl.getAccessForServer(revision, serverName);
        if (access.outcome !== AccessControl_1.Access.Allowed) {
            return Action_1.ActionError.Result(`The server ${serverName} has policies that are denying it access to protected rooms with the reason: ${(_b = (_a = access.rule) === null || _a === void 0 ? void 0 : _a.reason) !== null && _b !== void 0 ? _b : 'undefined'}`);
        }
        return await this.applyPolicyRevisionToSet(revisionIssuer);
    }
}
exports.ServerACLConequences = ServerACLConequences;
(0, CapabilityProvider_1.describeCapabilityProvider)({
    name: 'ServerACLConsequences',
    description: 'An implementation of ServerConsequences that uses m.room.server_acl to change access to rooms for servers.',
    interface: 'ServerConsequences',
    factory(_protectionDescription, context) {
        return new ServerACLConequences(context.stateEventSender, context.protectedRoomsSet);
    },
});
//# sourceMappingURL=ServerACLConsequences.js.map