import { StateEvent } from '../MatrixTypes/Events';
export declare enum StateChangeType {
    /**
     * A state event that has content has been introduced where no previous state type-key pair had
     * in the room's history. This also means where there are no previous redacted
     * or blanked state events.
     */
    Introduced = "Introduced",
    /**
     * A state event that has content has been reintroduced where a blank or redacted state type-key
     * pair had previously resided in the room state.
     * The distinction between introduced and reintroduced are important, because
     * an issuer can always treat introduced state in the timeline as a delta,
     * but not reintroduced, modified or removed state.
     */
    Reintroduced = "Reintroduced",
    /**
     * This is a special case of introduced, where a state type-key pair has been
     * introduced for the first time, but with empty content.
     */
    IntroducedAsBlank = "IntroducedAsBlank",
    /**
     * This is when a unique state event with empty content has been added
     * where there was previously a state event with empty or entirely redacted content.
     * Can alternatively be thought of as "ReintroducedAsEmpty".
     */
    BlankedEmptyContent = "BlankedEmptyContent",
    /**
     * A state event with empty content has been sent over a contentful event
     * with the same type-key pair.
     */
    BlankedContent = "BlankedContent",
    /**
     * A redaction was sent for an existing state event that is being tracked
     * and has removed all content keys.
     */
    CompletelyRedacted = "CompletelyRedacted",
    /**
     * A redaction was sent for an existing state event that is being tracked
     * and has removed all content keys that are not protected by authorization rules.
     * For example `membership` in a member event will not be removed.
     */
    PartiallyRedacted = "PartiallyRedacted",
    /**
     * There is an existing contentful state event for this type-key pair that has been replaced
     * with a new contenful state event.
     */
    SupersededContent = "SupersededContent",
    /**
     * The events are the same, and the event is intact.
     */
    NoChange = "NoChange"
}
/**
 * Calculate the change in the room state.
 * This is used on a per event basis on state deltas or the result of `/state`.
 * If calculating the effects of a redaction, apply the redaction first and then
 * compare the original event with the redacted version.
 * @param event A new event for a state type-key pair.
 * @param existingState Any known existing state event for the type-key pair.
 * @returns How the state was changed @see StateChangeType.
 */
export declare function calculateStateChange(event: StateEvent, existingState?: StateEvent): StateChangeType;
export declare function isChanged(changeType: StateChangeType): boolean;
export declare function isChangingContent(changeType: StateChangeType): boolean;
//# sourceMappingURL=StateChangeType.d.ts.map