"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
const matrix_protection_suite_1 = require("matrix-protection-suite");
const typescript_result_1 = require("@gnuxie/typescript-result");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const Rules_1 = require("../../commands/Rules");
function renderRestrictionOptions(options) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
        options.rule ? (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
            "rule: ",
            (0, Rules_1.renderRuleClearText)(options.rule))) : (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null)),
        interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
            "sender: ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, options.sender))));
}
function renderUserRestriction(userID, restriction, options) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
            "The resident user account ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, userID),
            " has been restricted (restriction type: ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, restriction),
            ")"),
        renderRestrictionOptions(options)));
}
class StandardUserRestrictionCapabilityRenderer {
    constructor(description, messageCollector, capability) {
        this.description = description;
        this.messageCollector = messageCollector;
        this.capability = capability;
        this.requiredEventPermissions = this.capability.requiredEventPermissions;
        this.requiredPermissions = this.capability.requiredPermissions;
        this.requiredStatePermissions = this.capability.requiredStatePermissions;
        this.isSimulated = Boolean(this.capability.isSimulated);
        // nothing to do.
    }
    async isUserRestricted(userID) {
        // nothing to render here since this doesn't do anything.
        return await this.capability.isUserRestricted(userID);
    }
    async restrictUser(userID, options) {
        const capabilityResult = await this.capability.restrictUser(userID, options);
        if ((0, typescript_result_1.isError)(capabilityResult)) {
            this.messageCollector.addOneliner(this.description, this.capability, (0, mps_interface_adaptor_1.renderFailedSingularConsequence)(this.description, interface_manager_1.DeadDocumentJSX.JSXFactory("span", null,
                "Failed to restrict the user ",
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, userID)), capabilityResult.error));
            return capabilityResult;
        }
        this.messageCollector.addOneliner(this.description, this.capability, renderUserRestriction(userID, capabilityResult.ok, options));
        return capabilityResult;
    }
    async unrestrictUser(userID, sender) {
        const capabilityResult = await this.capability.unrestrictUser(userID, sender);
        if ((0, typescript_result_1.isError)(capabilityResult)) {
            this.messageCollector.addOneliner(this.description, this.capability, (0, mps_interface_adaptor_1.renderFailedSingularConsequence)(this.description, interface_manager_1.DeadDocumentJSX.JSXFactory("span", null,
                "Failed to unrestrict the user ",
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, userID)), capabilityResult.error));
            return capabilityResult;
        }
        this.messageCollector.addOneliner(this.description, this.capability, interface_manager_1.DeadDocumentJSX.JSXFactory("span", null,
            "The user ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, userID),
            " has been unrestricted."));
        return capabilityResult;
    }
}
(0, matrix_protection_suite_1.describeCapabilityRenderer)({
    name: StandardUserRestrictionCapabilityRenderer.name,
    description: "Renders capabilities supporting the UserRestrictionCapability interface",
    interface: "UserRestrictionCapability",
    isDefaultForInterface: true,
    factory(protectionDescription, draupnir, capability) {
        return new StandardUserRestrictionCapabilityRenderer(protectionDescription, draupnir.capabilityMessageRenderer, capability);
    },
});
//# sourceMappingURL=UserRestrictionCapabilityRenderer.js.map