"use strict";
// Copyright (C) 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqliteRoomStateBackingStore = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const BetterSqliteStore_1 = require("./BetterSqliteStore");
const utils_1 = require("../../utils");
const path_1 = __importDefault(require("path"));
const SqliteSchema_1 = require("./SqliteSchema");
const log = new matrix_protection_suite_1.Logger("SqliteRoomStateBackingStore");
const SchemaText = [
    `
  CREATE TABLE room_info (
    room_id TEXT PRIMARY KEY NOT NULL,
    last_complete_writeback INTEGER NOT NULL
  ) STRICT, WITHOUT ROWID;
  CREATE TABLE room_state_event (
    room_id TEXT NOT NULL,
    event_type TEXT NOT NULL,
    state_key TEXT NOT NULL,
    event BLOB NOT NULL,
    PRIMARY KEY (room_id, event_type, state_key),
    FOREIGN KEY (room_id) REFERENCES room_info(room_id)
  ) STRICT;
  `,
];
const SchemaOptions = {
    upgradeSteps: SchemaText.map((text) => function (db) {
        db.exec(text);
    }),
    legacyUpgrade(db) {
        // An older version of this store used a different schema management system
        // fortunatley the data is just a psersistent cache so we don't need to worry about removing it.
        db.exec(`
      DROP TABLE room_state_event;
      DROP TABLE room_info;
      DROP TABLE schema;
    `);
    },
    consistencyCheck(db) {
        return (0, SqliteSchema_1.checkKnownTables)(db, ["room_info", "room_state_event"]);
    },
};
class SqliteRoomStateBackingStore extends BetterSqliteStore_1.BetterSqliteStore {
    constructor(options, db, eventDecoder) {
        super(SchemaOptions, db, log);
        this.eventDecoder = eventDecoder;
        this.roomInfoMap = new Map();
        this.revisionListener = this.handleRevision.bind(this);
    }
    static create(storagePath, eventDecoder) {
        const options = {
            path: path_1.default.join(storagePath, SqliteRoomStateBackingStore.StoreName),
            WALMode: true,
            foreignKeys: true,
            fileMustExist: false,
        };
        return new SqliteRoomStateBackingStore(options, (0, BetterSqliteStore_1.makeBetterSqliteDB)(options, log), eventDecoder);
    }
    updateBackingStore(revision, changes) {
        const roomMetaStatement = this.db.prepare(`REPLACE INTO room_info VALUES(?, ?)`);
        const replaceStatement = this.db.prepare(`REPLACE INTO room_state_event VALUES(?, ?, ?, jsonb(?))`);
        const createValue = (event) => {
            return [
                event.room_id,
                event.type,
                event.state_key,
                JSON.stringify(event),
            ];
        };
        // `flatTransaction` optimizes away unnecessary temporary files.
        const replace = (0, BetterSqliteStore_1.flatTransaction)(this.db, (events) => {
            for (const event of events) {
                replaceStatement.run(createValue(event));
            }
        });
        const doCompleteWriteback = this.db.transaction(() => {
            const info = {
                room_id: revision.room.toRoomIDOrAlias(),
                last_complete_writeback: Date.now(),
            };
            roomMetaStatement.run(info.room_id, info.last_complete_writeback);
            replace(revision.allState);
            this.roomInfoMap.set(info.room_id, info);
        });
        const roomInfo = this.getRoomMeta(revision.room.toRoomIDOrAlias());
        if (roomInfo === undefined) {
            try {
                doCompleteWriteback();
            }
            catch (e) {
                log.error(`Unable to create initial room state for ${revision.room.toPermalink()} into the room state backing store`, e);
            }
        }
        else {
            try {
                replace(changes.map((change) => change.state));
            }
            catch (e) {
                log.error(`Unable to update the room state for ${revision.room.toPermalink()} as a result of ${changes.length} changes`, e);
            }
        }
    }
    handleRevision(revision, changes) {
        try {
            this.updateBackingStore(revision, changes);
        }
        catch (e) {
            log.error(`Unable to update the backing store for revision of the room ${revision.room.toPermalink()}`, e);
        }
    }
    getRoomMeta(roomID) {
        const entry = this.roomInfoMap.get(roomID);
        if (entry) {
            return entry;
        }
        else {
            const dbEntry = this.db
                .prepare(`SELECT * FROM room_info WHERE room_id = ?`)
                .get(roomID);
            if (dbEntry === undefined) {
                return dbEntry;
            }
            this.roomInfoMap.set(roomID, dbEntry);
            return dbEntry;
        }
    }
    getRoomState(roomID) {
        const roomInfo = this.getRoomMeta(roomID);
        if (roomInfo === undefined) {
            return Promise.resolve((0, matrix_protection_suite_1.Ok)(undefined));
        }
        else {
            const events = [];
            for (const event of this.db
                .prepare(`SELECT json(event) FROM room_state_event WHERE room_id = ?`)
                .pluck()
                .iterate(roomID)) {
                const rawJson = JSON.parse(event, utils_1.jsonReviver);
                // We can't trust what's in the store, because our event decoders might have gotten
                // stricter in more recent versions. Meaning the store could have invalid events
                // that we don't want to blindly intern.
                const decodedEvent = this.eventDecoder.decodeStateEvent(rawJson);
                if ((0, matrix_protection_suite_1.isError)(decodedEvent)) {
                    log.error(`Unable to decode event from store:`, decodedEvent.error);
                    continue;
                }
                else {
                    events.push(decodedEvent.ok);
                }
            }
            return Promise.resolve((0, matrix_protection_suite_1.Ok)(events));
        }
    }
    forgetRoom(roomID) {
        const deleteStateStatement = this.db.prepare(`DELETE FROM room_state_event WHERE room_id = ?`);
        const deleteMetaStatement = this.db.prepare(`DELETE FROM room_info WHERE room_id = ?`);
        const deleteRoom = this.db.transaction(() => {
            deleteStateStatement.run(roomID);
            deleteMetaStatement.run(roomID); // needs to be last to avoid violating foriegn key cnostraint
        });
        try {
            deleteRoom();
        }
        catch (e) {
            return Promise.resolve(matrix_protection_suite_1.ActionException.Result(`Unable to forget the room ${roomID}`, {
                exception: e,
                exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown,
            }));
        }
        return Promise.resolve((0, matrix_protection_suite_1.Ok)(undefined));
    }
    async forgetAllRooms() {
        this.db.transaction(() => {
            this.db.exec(`
        DELETE FROM room_state_event;
        DELETE FROM room_info;
      `);
            this.roomInfoMap.clear();
        })();
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
}
exports.SqliteRoomStateBackingStore = SqliteRoomStateBackingStore;
SqliteRoomStateBackingStore.StoreName = "room-state-backing-store.db";
//# sourceMappingURL=SqliteRoomStateBackingStore.js.map