"use strict";
// Copyright 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirProtectionsShowCommand = void 0;
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const typescript_result_1 = require("@gnuxie/typescript-result");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
const PersistentConfigRenderer_1 = require("../safemode/PersistentConfigRenderer");
exports.DraupnirProtectionsShowCommand = (0, interface_manager_1.describeCommand)({
    summary: "Show a description of and the configured protection settings for a protection",
    parameters: (0, interface_manager_1.tuple)({
        name: "protection name",
        acceptor: interface_manager_1.StringPresentationType,
    }),
    async executor(draupnir, _info, _keywords, _rest, protectionName) {
        const protectionDescription = (0, matrix_protection_suite_1.findProtection)(protectionName);
        if (protectionDescription === undefined) {
            return typescript_result_1.ResultError.Result(`Cannot find a protection named ${protectionName}`);
        }
        const enabledProtections = draupnir.protectedRoomsSet.protections.allProtections;
        const protection = enabledProtections.find((protection) => protection.description === protectionDescription);
        const settings = await draupnir.protectedRoomsSet.protections.getProtectionSettings(protectionDescription);
        if ((0, typescript_result_1.isError)(settings)) {
            return settings.elaborate(`Unable to fetch the protection settings for the protection ${protectionName}`);
        }
        const capabilityProviderSet = await draupnir.protectedRoomsSet.protections.getCapabilityProviderSet(protectionDescription);
        if ((0, typescript_result_1.isError)(capabilityProviderSet)) {
            return capabilityProviderSet.elaborate(`Unable to fetch the capability provider set for the protection ${protectionName}`);
        }
        return (0, typescript_result_1.Ok)({
            description: protectionDescription,
            isEnabled: protection !== undefined,
            protection,
            config: settings.ok,
            activeCapabilityProviderSet: capabilityProviderSet.ok,
        });
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirProtectionsShowCommand, {
    JSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        const protectionInfo = commandResult.ok;
        return (0, typescript_result_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("span", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, protectionInfo.description.name),
                " ",
                protectionInfo.isEnabled ? "🟢 (enabled)" : "🔴 (disabled)"),
            interface_manager_1.DeadDocumentJSX.JSXFactory("p", null, protectionInfo.description.description),
            interface_manager_1.DeadDocumentJSX.JSXFactory("h3", null, "Protection settings"),
            PersistentConfigRenderer_1.StandardPersistentConfigRenderer.renderConfigStatus({
                description: protectionInfo.description.protectionSettings,
                error: undefined,
                data: protectionInfo.config,
            }),
            interface_manager_1.DeadDocumentJSX.JSXFactory("p", null,
                "To change a setting, use the",
                " ",
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null,
                    "!draupnir protections config ",
                    "<",
                    "add/set/remove",
                    ">",
                    " ",
                    protectionInfo.description.name,
                    " ",
                    "<",
                    "property name",
                    ">",
                    " ",
                    "<",
                    "value",
                    ">"),
                " ",
                "command. Protections may provide more conveniant commands to manage their settings."),
            interface_manager_1.DeadDocumentJSX.JSXFactory("h3", null, "Capability provider set"),
            Object.keys(protectionInfo.activeCapabilityProviderSet).length ===
                0 ? (interface_manager_1.DeadDocumentJSX.JSXFactory("p", null, "There are no configurable capabilities for this protection.")) : (renderCapabilityProviderSet(protectionInfo.activeCapabilityProviderSet))));
    },
});
function renderCapabilityProvider(name, capabilityProvider) {
    const compatibleProviders = (0, matrix_protection_suite_1.findCompatibleCapabilityProviders)(capabilityProvider.interface.name);
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
            "capability name: ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, name),
            ", interface:",
            " ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, capabilityProvider.interface.name),
            ", active capability provider: ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, capabilityProvider.name)),
        "interface description: ",
        capabilityProvider.interface.description,
        interface_manager_1.DeadDocumentJSX.JSXFactory("h4", null,
            "compatible capability providers for",
            " ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, capabilityProvider.interface.name),
            ":"),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, compatibleProviders.map((capability) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, capability.name),
            capability === capabilityProvider ? (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null, " (active)")) : (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null)),
            " ",
            "- ",
            capability.description))))));
}
function renderCapabilityProviderSet(set) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, Object.entries(set).map(([name, provider]) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null, renderCapabilityProvider(name, provider))))),
        "To change the active capability provider for a protection capability, use the",
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null,
            "!draupnir protections capability ",
            "<",
            "protection name",
            ">",
            " ",
            "<",
            "capability name",
            ">",
            " ",
            "<",
            "capability provider name",
            ">"),
        " ",
        "command. To reset the capability provider set to its default capabilities, use the",
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null,
            "!draupnir protections capability reset ",
            "<",
            "protection name",
            ">"),
        " ",
        "command."));
}
//# sourceMappingURL=ProtectionsShowCommand.js.map