"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirUnwatchPolicyRoomCommand = exports.DraupnirWatchPolicyRoomCommand = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const typescript_result_1 = require("@gnuxie/typescript-result");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
exports.DraupnirWatchPolicyRoomCommand = (0, interface_manager_1.describeCommand)({
    summary: "Watches a list and applies the list's assocated policies to draupnir's protected rooms.",
    parameters: (0, interface_manager_1.tuple)({
        name: "policy room",
        acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
    }),
    async executor({ watchedPolicyRooms, roomResolver }, _info, _keywords, _rest, policyRoomReference) {
        const policyRoom = await roomResolver.resolveRoom(policyRoomReference);
        if ((0, matrix_protection_suite_1.isError)(policyRoom)) {
            return policyRoom;
        }
        if (watchedPolicyRooms.allRooms.some((profile) => profile.room.toRoomIDOrAlias() === policyRoom.ok.toRoomIDOrAlias())) {
            return typescript_result_1.ResultError.Result("We are already watching this list.");
        }
        return await watchedPolicyRooms.watchPolicyRoomDirectly(policyRoom.ok);
    },
});
exports.DraupnirUnwatchPolicyRoomCommand = (0, interface_manager_1.describeCommand)({
    summary: "Unwatches a list and stops applying the list's assocated policies to draupnir's protected rooms.",
    parameters: (0, interface_manager_1.tuple)({
        name: "policy room",
        acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
    }),
    async executor({ watchedPolicyRooms, roomResolver }, _info, _keywords, _rest, policyRoomReference) {
        const policyRoom = await roomResolver.resolveRoom(policyRoomReference);
        if ((0, matrix_protection_suite_1.isError)(policyRoom)) {
            return policyRoom;
        }
        return await watchedPolicyRooms.unwatchPolicyRoom(policyRoom.ok);
    },
});
for (const command of [
    exports.DraupnirWatchPolicyRoomCommand,
    exports.DraupnirUnwatchPolicyRoomCommand,
]) {
    DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(command, {
        isAlwaysSupposedToUseDefaultRenderer: true,
    });
    DraupnirCommandPrerequisites_1.DraupnirContextToCommandContextTranslator.registerTranslation(command, (draupnir) => ({
        watchedPolicyRooms: draupnir.protectedRoomsSet.watchedPolicyRooms,
        roomResolver: draupnir.clientPlatform.toRoomResolver(),
    }));
}
//# sourceMappingURL=WatchUnwatchCommand.js.map