"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardMatrixInterfaceCommandDispatcher = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const StandardCommandDispatcher_1 = require("../TextReader/StandardCommandDispatcher");
class StandardMatrixInterfaceCommandDispatcher {
    constructor(interfaceAdaptor, adaptorContext, commandTable, helpCommand, invocationInformationFromEventContext, callbacks, 
    /**
     * Sometimes it is useful to check whether all commands in a table have a renderer, and all renderers have a command in the table.
     * Becuase people can forget to import them properly. This can be disabled if table imports are dynamic.
     */
    verifyOptions) {
        var _a, _b;
        this.interfaceAdaptor = interfaceAdaptor;
        this.adaptorContext = adaptorContext;
        this.commandTable = commandTable;
        this.helpCommand = helpCommand;
        this.invocationInformationFromEventContext = invocationInformationFromEventContext;
        this.baseDispatcher =
            new StandardCommandDispatcher_1.StandardCommandDispatcher(this.commandTable, this.helpCommand, callbacks);
        if ((_a = verifyOptions === null || verifyOptions === void 0 ? void 0 : verifyOptions.verifyRenderers) !== null && _a !== void 0 ? _a : true) {
            this.verifyAdaptorRenderingAllCommands();
        }
        if ((_b = verifyOptions === null || verifyOptions === void 0 ? void 0 : verifyOptions.verifyTable) !== null && _b !== void 0 ? _b : true) {
            this.verifyTableImportingAllRenderedCommands();
        }
    }
    handleCommandFromPresentationStream(eventContext, stream) {
        const partialCommand = this.baseDispatcher.parsePartialCommandFromStream(this.invocationInformationFromEventContext(eventContext), stream);
        if ((0, typescript_result_1.isError)(partialCommand)) {
            return; // callbacks should be handled by the baseDispatcher already.
        }
        void this.interfaceAdaptor.parseAndInvoke(partialCommand.ok, this.adaptorContext, eventContext);
    }
    handleCommandMessageEvent(eventContext, body) {
        const partialCommand = this.baseDispatcher.parsePartialCommandFromBody(this.invocationInformationFromEventContext(eventContext), body);
        if ((0, typescript_result_1.isError)(partialCommand)) {
            return; // callbacks should be handled by the baseDispatcher already.
        }
        void this.interfaceAdaptor.parseAndInvoke(partialCommand.ok, this.adaptorContext, eventContext);
    }
    verifyAdaptorRenderingAllCommands() {
        for (const command of this.commandTable.getAllCommands()) {
            if (!this.interfaceAdaptor.isDescribingRendererForCommand(command.currentCommand)) {
                throw new TypeError(`Adaptor does not render command ${command.designator.toString()}`);
            }
        }
    }
    verifyTableImportingAllRenderedCommands() {
        for (const command of this.interfaceAdaptor.renderedCommands()) {
            if (!this.commandTable.isContainingCommand(command)) {
                throw new TypeError(`Command table does not contain a command that is specified in the interface adaptor ${command.summary}\n${command.description}`);
            }
        }
    }
}
exports.StandardMatrixInterfaceCommandDispatcher = StandardMatrixInterfaceCommandDispatcher;
//# sourceMappingURL=MatrixInterfaceCommandDispatcher.js.map