"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardDirectPropagationPolicyListRevisionIssuer = void 0;
// Copyright 2022 - 2023 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
const events_1 = require("events");
const StandardPolicyListRevision_1 = require("../PolicyList/StandardPolicyListRevision");
const PolicyRuleChange_1 = require("../PolicyList/PolicyRuleChange");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
class StandardDirectPropagationPolicyListRevisionIssuer extends events_1.EventEmitter {
    constructor(issuers, filter) {
        super();
        this.filter = filter;
        this.revision = StandardPolicyListRevision_1.StandardPolicyListRevision.blankRevision();
        this.revisionListener = this.handleRevision.bind(this);
        this.policyListRevisionIssuers = new Set();
        this.addIssuers(issuers);
    }
    filterChanges(changes) {
        const filter = this.filter; // narrowing isn't working for some reason.
        return filter ? changes.filter((change) => filter(change)) : changes;
    }
    handleRevision(_newRevision, unfilteredChanges) {
        const changes = this.filterChanges(unfilteredChanges);
        if (changes.length > 0) {
            const oldRevision = this.revision;
            this.revision = this.revision.reviseFromChanges(this.filterChanges(changes));
            this.emit('revision', this.revision, changes, oldRevision);
        }
    }
    get currentRevision() {
        return this.revision;
    }
    unregisterListeners() {
        for (const issuer of this.policyListRevisionIssuers) {
            issuer.off('revision', this.revisionListener);
        }
    }
    get references() {
        const references = [];
        for (const issuer of this.policyListRevisionIssuers) {
            // i don't like this adhoc structural typing, but we don't really have a choice.
            if ('room' in issuer && issuer.room instanceof matrix_basic_types_1.MatrixRoomID) {
                references.push(issuer.room);
            }
        }
        return references;
    }
    previewIncorperationOfRevision(revision) {
        const changes = [];
        for (const policy of revision.allRules()) {
            changes.push({
                changeType: PolicyRuleChange_1.PolicyRuleChangeType.Added,
                event: policy.sourceEvent,
                sender: policy.sourceEvent.sender,
                rule: policy,
            });
        }
        return changes;
    }
    previewRemovalOfRevision(revision) {
        const changes = [];
        for (const policy of revision.allRules()) {
            changes.push({
                changeType: PolicyRuleChange_1.PolicyRuleChangeType.Removed,
                event: policy.sourceEvent,
                sender: policy.sourceEvent.sender,
                rule: policy,
                previousRule: policy,
            });
        }
        return changes;
    }
    addIssuers(issuers) {
        let changes = [];
        for (const issuer of issuers) {
            this.policyListRevisionIssuers.add(issuer);
            issuer.on('revision', this.revisionListener);
            changes = changes.concat(this.previewIncorperationOfRevision(issuer.currentRevision));
        }
        this.handleRevision(this.revision, changes);
    }
    addIssuer(issuer) {
        this.addIssuers([issuer]);
    }
    removeIssuer(issuer) {
        issuer.off('revision', this.revisionListener);
        this.policyListRevisionIssuers.delete(issuer);
        const changes = this.previewRemovalOfRevision(issuer.currentRevision);
        this.handleRevision(this.revision, changes);
    }
}
exports.StandardDirectPropagationPolicyListRevisionIssuer = StandardDirectPropagationPolicyListRevisionIssuer;
//# sourceMappingURL=DirectPropagationPolicyListRevisionIssuer.js.map