"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.MessageIsVoiceProtection = void 0;
const matrix_bot_sdk_1 = require("matrix-bot-sdk");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
(0, matrix_protection_suite_1.describeProtection)({
    name: "MessageIsVoiceProtection",
    description: "If a user posts a voice message, that message will be redacted",
    capabilityInterfaces: {
        eventConsequences: "EventConsequences",
    },
    defaultCapabilities: {
        eventConsequences: "StandardEventConsequences",
    },
    factory: async function (description, protectedRoomsSet, draupnir, capabilities, _settings) {
        return (0, matrix_protection_suite_1.Ok)(new MessageIsVoiceProtection(description, capabilities, protectedRoomsSet, draupnir));
    },
});
class MessageIsVoiceProtection extends matrix_protection_suite_1.AbstractProtection {
    constructor(description, capabilities, protectedRoomsSet, draupnir) {
        super(description, capabilities, protectedRoomsSet, {});
        this.draupnir = draupnir;
        this.eventConsequences = capabilities.eventConsequences;
    }
    async handleTimelineEvent(room, event) {
        const roomID = room.toRoomIDOrAlias();
        if (matrix_protection_suite_1.Value.Check(matrix_protection_suite_1.RoomMessage, event)) {
            if (!("msgtype" in event.content) ||
                event.content.msgtype !== "m.audio") {
                return (0, matrix_protection_suite_1.Ok)(undefined);
            }
            await this.draupnir.managementRoomOutput.logMessage(matrix_bot_sdk_1.LogLevel.INFO, "MessageIsVoice", `Redacting event from ${event["sender"]} for posting a voice message. ${matrix_basic_types_1.Permalinks.forEvent(roomID, event["event_id"], [(0, matrix_basic_types_1.userServerName)(this.draupnir.clientUserID)])}`);
            // Redact the event
            if (!this.draupnir.config.noop) {
                return await this.eventConsequences.consequenceForEvent(roomID, event["event_id"], "Voice messages are not permitted here");
            }
            else {
                await this.draupnir.managementRoomOutput.logMessage(matrix_bot_sdk_1.LogLevel.WARN, "MessageIsVoice", `Tried to redact ${event["event_id"]} in ${roomID} but Draupnir is running in no-op mode`, roomID);
                return (0, matrix_protection_suite_1.Ok)(undefined);
            }
        }
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
}
exports.MessageIsVoiceProtection = MessageIsVoiceProtection;
//# sourceMappingURL=MessageIsVoice.js.map