"use strict";
// Copyright 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.NewJoinerProtection = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
class NewJoinerProtection extends matrix_protection_suite_1.AbstractProtection {
    constructor(description, capabilities, protectedRoomsSet, draupnirConfig) {
        super(description, capabilities, protectedRoomsSet, {});
        this.userConsequences = capabilities.userConsequences;
        this.bannedServers = new Set(draupnirConfig.protections.newJoinerProtection.serverNames);
        this.banReason = draupnirConfig.protections.newJoinerProtection.banMessage;
    }
    async handleMembershipChange(revision, changes) {
        const errors = [];
        for (const change of changes) {
            if (change.membershipChangeType === matrix_protection_suite_1.MembershipChangeType.Joined) {
                if (this.bannedServers.has((0, matrix_basic_types_1.userServerName)(change.userID))) {
                    const banResult = await this.userConsequences.consequenceForUserInRoom(revision.room.toRoomIDOrAlias(), change.userID, this.banReason);
                    if ((0, matrix_protection_suite_1.isError)(banResult)) {
                        errors.push(banResult.error);
                    }
                }
            }
        }
        if (errors.length === 0) {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        else {
            return matrix_protection_suite_1.MultipleErrors.Result(`There were errors when banning members in ${revision.room.toPermalink()}`, { errors });
        }
    }
}
exports.NewJoinerProtection = NewJoinerProtection;
(0, matrix_protection_suite_1.describeProtection)({
    name: "NewJoinerProtection",
    description: `Highly experimental protection that will ban all new joiners from configured homeservers.
    Will not ban existing users from those servers, and unbanning users will allow them to join normally.
    Please read the documentation https://the-draupnir-project.github.io/draupnir-documentation/protections/new-joiner-protection.`,
    capabilityInterfaces: {
        userConsequences: "UserConsequences",
    },
    defaultCapabilities: {
        userConsequences: "StandardUserConsequences",
    },
    factory: async (decription, protectedRoomsSet, draupnir, capabilitySet) => (0, matrix_protection_suite_1.Ok)(new NewJoinerProtection(decription, capabilitySet, protectedRoomsSet, draupnir.config)),
});
//# sourceMappingURL=NewJoinerProtection.js.map