"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessControl = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
/**
 * Utility to manage which users have access to the application service,
 * meaning whether a user is able to provision a draupnir or continue to use one.
 * Internally we use a policy list within matrix to determine who has access via the `AccessControlUnit`.
 */
class AccessControl {
    constructor(accessControlRevisionIssuer, editor) {
        this.accessControlRevisionIssuer = accessControlRevisionIssuer;
        this.editor = editor;
        // nothing to do.
    }
    /**
     * Construct and initialize access control for the `MjolnirAppService`.
     * @param accessControlListId The room id of a policy list used to manage access to the appservice (who can provision & use draupnir)
     * @param bridge The matrix-appservice-bridge, used to get the appservice bot.
     * @returns A new instance of `AccessControl` to be used by `MjolnirAppService`.
     */
    static async setupAccessControlForRoom(
    /** The room id for the access control list. */
    accessControlRoom, policyRoomManager, bridgeBotJoiner) {
        const joinResult = await bridgeBotJoiner.joinRoom(accessControlRoom.toRoomIDOrAlias());
        if ((0, matrix_protection_suite_1.isError)(joinResult)) {
            return joinResult;
        }
        const revisionIssuer = await policyRoomManager.getPolicyRoomRevisionIssuer(accessControlRoom);
        if ((0, matrix_protection_suite_1.isError)(revisionIssuer)) {
            return revisionIssuer;
        }
        const editor = await policyRoomManager.getPolicyRoomEditor(accessControlRoom);
        if ((0, matrix_protection_suite_1.isError)(editor)) {
            return editor;
        }
        return (0, matrix_protection_suite_1.Ok)(new AccessControl(revisionIssuer.ok, editor.ok));
    }
    getUserAccess(mxid) {
        return matrix_protection_suite_1.AccessControl.getAccessForUser(this.accessControlRevisionIssuer.currentRevision, mxid, "CHECK_SERVER");
    }
    async allow(mxid, reason = "<no reason supplied>") {
        const result = await this.editor.createPolicy(matrix_protection_suite_1.PolicyRuleType.User, matrix_protection_suite_1.Recommendation.Allow, mxid, reason, {});
        if ((0, matrix_protection_suite_1.isError)(result)) {
            return result;
        }
        else {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
    }
    async remove(mxid) {
        const result = await this.editor.unbanEntity(matrix_protection_suite_1.PolicyRuleType.User, mxid);
        if ((0, matrix_protection_suite_1.isError)(result)) {
            return result;
        }
        else {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
    }
}
exports.AccessControl = AccessControl;
//# sourceMappingURL=AccessControl.js.map