"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirImportCommand = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_protection_suite_for_matrix_bot_sdk_1 = require("matrix-protection-suite-for-matrix-bot-sdk");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
exports.DraupnirImportCommand = (0, interface_manager_1.describeCommand)({
    summary: "Import user and server bans from a Matrix room and add them to a policy room.",
    parameters: (0, interface_manager_1.tuple)({
        name: "import from room",
        acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
    }, {
        name: "policy room",
        acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
    }),
    async executor(draupnir, _info, _keywords, _rest, importFromRoomReference, policyRoomReference) {
        const importFromRoom = await (0, matrix_protection_suite_for_matrix_bot_sdk_1.resolveRoomReferenceSafe)(draupnir.client, importFromRoomReference);
        if ((0, matrix_protection_suite_1.isError)(importFromRoom)) {
            return importFromRoom;
        }
        const policyRoom = await (0, matrix_protection_suite_for_matrix_bot_sdk_1.resolveRoomReferenceSafe)(draupnir.client, policyRoomReference);
        if ((0, matrix_protection_suite_1.isError)(policyRoom)) {
            return policyRoom;
        }
        const policyRoomEditor = await draupnir.policyRoomManager.getPolicyRoomEditor(policyRoom.ok);
        if ((0, matrix_protection_suite_1.isError)(policyRoomEditor)) {
            return policyRoomEditor;
        }
        const state = await draupnir.client.getRoomState(importFromRoom.ok.toRoomIDOrAlias());
        const errors = [];
        for (const stateEvent of state) {
            const content = stateEvent["content"] || {};
            if (!content || Object.keys(content).length === 0)
                continue;
            if (stateEvent["type"] === "m.room.member" &&
                stateEvent["state_key"] !== "") {
                // Member event - check for ban
                if (content["membership"] === "ban") {
                    const reason = content["reason"] || "<no reason>";
                    const result = await policyRoomEditor.ok.banEntity(matrix_protection_suite_1.PolicyRuleType.User, stateEvent["state_key"], reason);
                    if ((0, matrix_protection_suite_1.isError)(result)) {
                        errors.push(matrix_protection_suite_1.RoomActionError.fromActionError(policyRoom.ok, result.error));
                    }
                }
            }
            else if (stateEvent["type"] === "m.room.server_acl" &&
                stateEvent["state_key"] === "") {
                // ACL event - ban denied servers
                if (!content["deny"])
                    continue;
                for (const server of content["deny"]) {
                    const reason = "<no reason>";
                    const result = await policyRoomEditor.ok.banEntity(matrix_protection_suite_1.PolicyRuleType.Server, server, reason);
                    if ((0, matrix_protection_suite_1.isError)(result)) {
                        errors.push(matrix_protection_suite_1.RoomActionError.fromActionError(policyRoom.ok, result.error));
                    }
                }
            }
        }
        return matrix_protection_suite_1.MultipleErrors.Result(`There were multiple errors when importing bans from the room ${importFromRoomReference.toPermalink()} to ${policyRoomReference.toPermalink()}`, { errors });
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirImportCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
});
//# sourceMappingURL=ImportCommand.js.map