"use strict";
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynapseAdminDeactivateCommand = void 0;
exports.sendPromptDeactivation = sendPromptDeactivation;
const typescript_result_1 = require("@gnuxie/typescript-result");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const DraupnirCommandPrerequisites_1 = require("../DraupnirCommandPrerequisites");
const deactivateUser_1 = require("../../protections/HomeserverUserPolicyApplication/deactivateUser");
const Rules_1 = require("../Rules");
exports.SynapseAdminDeactivateCommand = (0, interface_manager_1.describeCommand)({
    summary: "Deactivate a user on the homeserver.",
    parameters: (0, interface_manager_1.tuple)({
        name: "user",
        description: "The user to deactivate",
        acceptor: interface_manager_1.MatrixUserIDPresentationType,
    }),
    keywords: {
        keywordDescriptions: {
            "purge-messages": {
                isFlag: true,
                description: "Restrict access to the account until Draupnir removes all of their messages, and then finally deactivate.",
            },
            "no-confirm": {
                isFlag: true,
                description: "Runs the command without the preview of the unban and the confirmation prompt.",
            },
        },
    },
    async executor(draupnir, info, keywords, _rest, targetUser) {
        const synapseAdminClient = draupnir.synapseAdminClient;
        const isAdmin = await synapseAdminClient?.isSynapseAdmin();
        if (isAdmin === undefined ||
            (0, typescript_result_1.isError)(isAdmin) ||
            !isAdmin.ok ||
            !draupnir.purgingDeactivate) {
            return matrix_protection_suite_1.ActionError.Result("I am not a Synapse administrator, or the endpoint to deactivate a user is blocked");
        }
        if (draupnir.stores.userRestrictionAuditLog === undefined) {
            return typescript_result_1.ResultError.Result("The user restriction audit log is not configured");
        }
        if (synapseAdminClient === undefined) {
            throw new TypeError("Shouldn't be happening at this point");
        }
        const isConfirmed = keywords.getKeywordValue("no-confirm", false);
        const isPurgingMessages = keywords.getKeywordValue("purge-messages", false);
        const previewResult = await (async () => {
            const details = await synapseAdminClient.getUserDetails(targetUser.toString());
            if ((0, typescript_result_1.isError)(details)) {
                return details.elaborate(`Failed to get details for the user ${targetUser.toString()}`);
            }
            else if (!details.ok) {
                return typescript_result_1.ResultError.Result(`Couldn't find a residident user with the ID ${targetUser.toString()}`);
            }
            return (0, typescript_result_1.Ok)({
                targetUser: targetUser.toString(),
                creation_timestamp: details.ok.creation_ts,
                displayname: details.ok.displayname ?? undefined,
                isPurgingMessages: Boolean(isPurgingMessages),
                isNoConfirm: Boolean(isConfirmed),
            });
        })();
        if (!isConfirmed) {
            return previewResult;
        }
        const deactivateResult = await (() => isPurgingMessages
            ? draupnir.purgingDeactivate.beginPurgeUser(targetUser.toString(), {
                sender: info.commandSender,
                rule: null,
            })
            : (0, deactivateUser_1.deactivateUser)(targetUser.toString(), synapseAdminClient, draupnir.stores.userRestrictionAuditLog, {
                sender: info.commandSender,
                rule: null,
            }))();
        if ((0, typescript_result_1.isError)(deactivateResult)) {
            return deactivateResult.elaborate(`Failed to deactivate the user ${targetUser.toString()}`);
        }
        else {
            return previewResult;
        }
    },
});
function renderUserDetails(preview) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
            "Details for ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, preview.targetUser)),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                "Creation date:",
                " ",
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, new Date(preview.creation_timestamp).toLocaleDateString())),
            interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                "Displayname:",
                " ",
                preview.displayname ? (interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, preview.displayname)) : ("None set")),
            interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                "Purging messages:",
                " ",
                preview.isPurgingMessages ? interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "Yes") : interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "No")))));
}
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.SynapseAdminDeactivateCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
    confirmationPromptJSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        const preview = commandResult.ok;
        return (0, typescript_result_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
            "You are about to deactivate the user ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, preview.targetUser),
            ". This will permanently deactivate the user and remove their access to the homeserver.",
            preview.isPurgingMessages ? (interface_manager_1.DeadDocumentJSX.JSXFactory("span", null, "Purging their messages will also cause their account to be used by the homeserver to send hundreds of redaction events to remove everything they have sent.")) : (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null)),
            renderUserDetails(preview)));
    },
    JSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        return (0, typescript_result_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
            "Successfully deactivated ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, commandResult.ok.targetUser),
            renderUserDetails(commandResult.ok)));
    },
});
function renderPromptDeactivation(targetUserID, policyRule) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
        "The resident user ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, targetUserID),
        " has been matched by a policy rule that would cause their messages to be redacted if they joined a protected room. Would you like to deactivate this user account and remove all their messages?",
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, (0, Rules_1.renderRuleSummary)(policyRule))));
}
// FIXME:
// hmm now would be a great time to include that feature where we can
// run commands. but there probably isn't time for that right now.
// we will just have to send a message, and then the confirmation prompt.
function sendPromptDeactivation(targetUserID, policyRule, managementRoomID, sendConfirmationPrompt, log) {
    void (0, matrix_protection_suite_1.Task)((async () => {
        const confirmationPromptResult = await sendConfirmationPrompt({
            commandDesignator: ["draupnir", "deactivate"],
            readItems: [targetUserID, "--purge-messages"],
        }, renderPromptDeactivation(targetUserID, policyRule), { roomID: managementRoomID });
        if ((0, typescript_result_1.isError)(confirmationPromptResult)) {
            log.error(`Failed to send deactivation confirmation prompt for`, targetUserID, confirmationPromptResult);
        }
    })());
}
//# sourceMappingURL=DeactivateCommand.js.map