"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2020 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynapseAdminShutdownRoomCommand = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const DraupnirCommandPrerequisites_1 = require("../DraupnirCommandPrerequisites");
exports.SynapseAdminShutdownRoomCommand = (0, interface_manager_1.describeCommand)({
    summary: "Prevents access to the the room on this server and sends a message to all users that they have violated the terms of service.",
    parameters: (0, interface_manager_1.tuple)({
        name: "room",
        acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
    }),
    rest: {
        name: "reason",
        acceptor: interface_manager_1.StringPresentationType,
    },
    keywords: {
        keywordDescriptions: {
            notify: {
                isFlag: true,
                description: "Whether to send the content violation notification message. This is an invitation that will be sent to the room's joined members which explains the reason for the shutdown.",
            },
        },
    },
    async executor(draupnir, _info, keywords, reasonParts, targetRoom) {
        const notify = keywords.getKeywordValue("notify", false);
        const isAdmin = await draupnir.synapseAdminClient?.isSynapseAdmin();
        if (isAdmin === undefined || (0, typescript_result_1.isError)(isAdmin) || !isAdmin.ok) {
            return matrix_protection_suite_1.ActionError.Result("I am not a Synapse administrator, or the endpoint to shutdown a room is blocked");
        }
        if (draupnir.synapseAdminClient === undefined) {
            throw new TypeError(`Should be impossible at this point.`);
        }
        const resolvedRoom = await draupnir.clientPlatform
            .toRoomResolver()
            .resolveRoom(targetRoom);
        if ((0, typescript_result_1.isError)(resolvedRoom)) {
            return resolvedRoom;
        }
        const reason = reasonParts.join(" ");
        return await draupnir.synapseAdminClient.shutdownRoomV2(resolvedRoom.ok.toRoomIDOrAlias(), {
            ...(notify
                ? { message: reason, new_room_user_id: draupnir.clientUserID }
                : {}),
            block: true,
            purge: true,
        });
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.SynapseAdminShutdownRoomCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
});
//# sourceMappingURL=ShutdownRoomCommand.js.map