"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019, 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DOCUMENTATION_URL = exports.SOFTWARE_VERSION = exports.PACKAGE_JSON = void 0;
exports.getNonDefaultConfigProperties = getNonDefaultConfigProperties;
exports.getDefaultConfig = getDefaultConfig;
exports.configRead = configRead;
exports.getProvisionedMjolnirConfig = getProvisionedMjolnirConfig;
exports.getUnknownPropertiesHelper = getUnknownPropertiesHelper;
exports.getUnknownConfigPropertyPaths = getUnknownConfigPropertyPaths;
exports.getStoragePath = getStoragePath;
const fs = __importStar(require("fs"));
const js_yaml_1 = require("js-yaml");
const matrix_bot_sdk_1 = require("matrix-bot-sdk");
// Needed for https://github.com/the-draupnir-project/Draupnir/issues/480
// sorry buddy...
process.env.SUPPRESS_NO_CONFIG_WARNING = "y";
const config_1 = __importDefault(require("config"));
const path_1 = __importDefault(require("path"));
const BootOption_1 = require("./safemode/BootOption");
const matrix_protection_suite_1 = require("matrix-protection-suite");
matrix_bot_sdk_1.LogService.setLogger(new matrix_bot_sdk_1.RichConsoleLogger());
(0, matrix_protection_suite_1.setGlobalLoggerProvider)(new matrix_bot_sdk_1.RichConsoleLogger());
const log = new matrix_protection_suite_1.Logger("Draupnir config");
/**
 * The version of the configuration that has been explicitly provided,
 * and does not contain default values. Secrets are marked with "REDACTED".
 */
function getNonDefaultConfigProperties(config) {
    const nonDefault = config_1.default.util.diffDeep(defaultConfig, config);
    if ("accessToken" in nonDefault) {
        nonDefault.accessToken = "REDACTED";
    }
    if ("pantalaimon" in nonDefault &&
        typeof nonDefault.pantalaimon === "object") {
        nonDefault.pantalaimon.password = "REDACTED";
    }
    if ("web" in nonDefault &&
        typeof nonDefault["web"] === "object" &&
        nonDefault["web"] !== null &&
        "synapseHTTPAntispam" in nonDefault["web"] &&
        typeof nonDefault["web"]["synapseHTTPAntispam"] === "object") {
        if (nonDefault["web"]["synapseHTTPAntispam"] !== null) {
            nonDefault["web"]["synapseHTTPAntispam"].authorization = "REDACTED";
        }
    }
    return nonDefault;
}
const defaultConfig = {
    homeserverUrl: "http://localhost:8008",
    rawHomeserverUrl: "http://localhost:8008",
    accessToken: "NONE_PROVIDED",
    pantalaimon: {
        use: false,
        username: "",
        password: "",
    },
    dataPath: "/data/storage",
    acceptInvitesFromSpace: "!noop:example.org",
    autojoinOnlyIfManager: true,
    recordIgnoredInvites: false,
    managementRoom: "!noop:example.org",
    logLevel: "INFO",
    logMutedModules: ["MatrixHttpClient", "MatrixClientLite"],
    verifyPermissionsOnStartup: true,
    noop: false,
    disableServerACL: false,
    automaticallyRedactForReasons: ["spam", "advertising"],
    protectAllJoinedRooms: false,
    backgroundDelayMS: 500,
    pollReports: false,
    displayReports: true,
    commands: {
        allowNoPrefix: false,
        symbolPrefixes: ["!"],
        additionalPrefixes: ["draupnir"],
        features: ["synapse admin"],
        ban: {
            defaultReasons: ["spam", "brigading", "harassment", "disagreement"],
        },
    },
    protections: {
        wordlist: {
            words: [],
            minutesBeforeTrusting: 20,
        },
        newJoinerProtection: {
            serverNames: [],
            banMessage: "Unfortunately we cannot accept new users from your homeserver at this time.",
        },
    },
    safeMode: {
        bootOption: BootOption_1.SafeModeBootOption.RecoveryOnly,
    },
    health: {
        healthz: {
            enabled: false,
            port: 8080,
            address: "0.0.0.0",
            endpoint: "/healthz",
            healthyStatus: 200,
            unhealthyStatus: 418,
        },
        sentry: undefined,
    },
    admin: {
        enableMakeRoomAdminCommand: false,
    },
    web: {
        enabled: false,
        port: 8080,
        address: "localhost",
        abuseReporting: {
            enabled: false,
        },
        synapseHTTPAntispam: {
            enabled: false,
            authorization: "DEFAULT",
        },
    },
    roomStateBackingStore: {
        enabled: true,
    },
    experimentalRustCrypto: false,
    draupnirNewsURL: "https://raw.githubusercontent.com/the-draupnir-project/Draupnir/refs/heads/main/src/protections/DraupnirNews/news.json",
    configMeta: undefined,
};
function getDefaultConfig() {
    return config_1.default.util.cloneDeep(defaultConfig);
}
function logNonDefaultConfiguration(config) {
    log.info("non-default configuration properties:", JSON.stringify(getNonDefaultConfigProperties(config), null, 2));
}
function logConfigMeta(config) {
    log.info("Configuration meta:", JSON.stringify(config.configMeta, null, 2));
}
function getConfigPath() {
    const draupnirPath = getCommandLineOption(process.argv, "--draupnir-config");
    if (draupnirPath) {
        return { isDraupnirPath: true, path: draupnirPath };
    }
    const mjolnirPath = getCommandLineOption(process.argv, "--mjolnir-config");
    if (mjolnirPath) {
        return { isDraupnirPath: false, path: mjolnirPath };
    }
    const path = config_1.default.util.getConfigSources().at(-1)?.name;
    if (path === undefined) {
        throw new TypeError("No configuration path has been found for Draupnir. Use the --draupnir-config option to provide a path to the config.");
    }
    return { isDraupnirPath: false, path };
}
function getConfigMeta() {
    const { isDraupnirPath, path } = getConfigPath();
    return {
        configPath: path,
        isDraupnirConfigOptionUsed: isDraupnirPath,
        isAccessTokenPathOptionUsed: isCommandLineOptionPresent(process.argv, "--access-token-path"),
        isPasswordPathOptionUsed: isCommandLineOptionPresent(process.argv, "--pantalaimon-password-path"),
        isHttpAntispamAuthorizationPathOptionUsed: isCommandLineOptionPresent(process.argv, "--http-antispam-authorization-path"),
    };
}
let configHookExitDump = undefined;
function registerConfigExitHook(config) {
    if (configHookExitDump) {
        return;
    }
    configHookExitDump = () => {
        logNonDefaultConfiguration(config);
        logConfigMeta(config);
    };
    process.on("exit", configHookExitDump);
}
/**
 * @returns The users's raw config, deep copied over the `defaultConfig`.
 */
function readConfigSource() {
    const configMeta = getConfigMeta();
    const config = (() => {
        const content = fs.readFileSync(configMeta.configPath, "utf8");
        const parsed = (0, js_yaml_1.load)(content);
        return config_1.default.util.extendDeep({}, defaultConfig, parsed, {
            configMeta: configMeta,
        });
    })();
    logConfigMeta(config);
    if (!configMeta.isDraupnirConfigOptionUsed) {
        log.warn("DEPRECATED", "Starting Draupnir without the --draupnir-config option is deprecated. Please provide Draupnir's configuration explicitly with --draupnir-config.", "config path used:", config.configMeta?.configPath);
    }
    const unknownProperties = getUnknownConfigPropertyPaths(config);
    if (unknownProperties.length > 0) {
        log.warn("There are unknown configuration properties, possibly a result of typos:", unknownProperties);
    }
    registerConfigExitHook(config);
    return config;
}
function configRead() {
    const config = readConfigSource();
    const explicitAccessTokenPath = getCommandLineOption(process.argv, "--access-token-path");
    const explicitPantalaimonPasswordPath = getCommandLineOption(process.argv, "--pantalaimon-password-path");
    const explicitHttpAntispamAuthorizationPath = getCommandLineOption(process.argv, "--http-antispam-authorization-path");
    if (explicitAccessTokenPath !== undefined) {
        config.accessToken = readSecretFromPath(explicitAccessTokenPath);
    }
    if (explicitPantalaimonPasswordPath) {
        config.pantalaimon.password = readSecretFromPath(explicitPantalaimonPasswordPath);
    }
    if (explicitHttpAntispamAuthorizationPath) {
        config.web.synapseHTTPAntispam.authorization = readSecretFromPath(explicitHttpAntispamAuthorizationPath);
    }
    return config;
}
/**
 * Provides a config for each newly provisioned draupnir in appservice mode.
 * @param managementRoomId A room that has been created to serve as the draupnir's management room for the owner.
 * @returns A config that can be directly used by the new draupnir.
 */
function getProvisionedMjolnirConfig(managementRoomId) {
    // These are keys that are allowed to be configured for provisioned draupnirs.
    // We need a restricted set so that someone doesn't accidentally enable webservers etc
    // on every created Draupnir, which would result in very confusing error messages.
    const allowedKeys = [
        "commands",
        "logLevel",
        "verifyPermissionsOnStartup",
        "automaticallyRedactForReasons",
        "protectAllJoinedRooms",
        "backgroundDelayMS",
        "safeMode",
    ];
    const configTemplate = configRead(); // we use the standard bot config as a template for every provisioned draupnir.
    const unusedKeys = Object.keys(configTemplate).filter((key) => !allowedKeys.includes(key));
    if (unusedKeys.length > 0) {
        matrix_bot_sdk_1.LogService.warn("config", "The config provided for provisioned draupnirs contains keys which are not used by the appservice.", unusedKeys);
    }
    const config = config_1.default.util.extendDeep(getDefaultConfig(), allowedKeys.reduce((existingConfig, key) => {
        return { ...existingConfig, [key]: configTemplate[key] };
    }, {}));
    config.managementRoom = managementRoomId;
    return config;
}
exports.PACKAGE_JSON = (() => {
    try {
        return JSON.parse(fs.readFileSync(path_1.default.join(__dirname, "../package.json"), "utf-8"));
    }
    catch (e) {
        matrix_bot_sdk_1.LogService.error("config", "Could not read Draupnir package.json", e);
        return {};
    }
})();
exports.SOFTWARE_VERSION = (() => {
    let versionFile;
    const defaultText = exports.PACKAGE_JSON.version ??
        "A version was either not provided when building Draupnir or could not be read.";
    try {
        versionFile = fs.readFileSync(path_1.default.join(__dirname, "../version.txt"), "utf-8");
    }
    catch (e) {
        matrix_bot_sdk_1.LogService.error("config", "Could not read Draupnir version", e);
        versionFile = defaultText;
    }
    // it's important to ignore the newline if the version is going to be put
    // into <pre> or <code> where it will create an unnecessary newline.
    return /^(.*)$/m.exec(versionFile)?.at(0) ?? defaultText;
})();
exports.DOCUMENTATION_URL = "https://the-draupnir-project.github.io/draupnir-documentation/";
// Command line related functions
/**
 * Grabs an option from the command line and checks if it exists.
 * @param args Program options
 * @param optionName Option name
 * @returns True if the option is present, otherwise false.
 */
function isCommandLineOptionPresent(args, optionName) {
    return args.includes(optionName);
}
/**
 * Grabs an option's value from program options if it exists, otherwise returns undefined.
 * @param args Program options
 * @param optionName Option name
 * @returns The value passed to the option, or undefined if the option is not specified.
 * @throws Error if the option is present but has no value.
 */
function getCommandLineOption(args, optionName) {
    // We don't want to throw if the option is not present
    if (!isCommandLineOptionPresent(args, optionName)) {
        return undefined;
    }
    const optionIndex = args.indexOf(optionName);
    //check if the next index is not an option
    const associatedArgument = args[optionIndex + 1];
    if (associatedArgument !== undefined &&
        !associatedArgument.startsWith("--")) {
        return associatedArgument;
    }
    // No value was provided, or the next argument is another option
    throw new Error(`No value provided for ${optionName}`);
}
function readSecretFromPath(path) {
    // extract only the first line.
    const secret = fs.readFileSync(path, "utf8").match(/^[^\r\n]*/)?.[0];
    if (!secret) {
        throw new TypeError(`There is no secret present in the file at ${path}`);
    }
    return secret;
}
function getUnknownPropertiesHelper(rawConfig, rawDefaults, currentPathProperties) {
    const unknownProperties = [];
    if (typeof rawConfig !== "object" ||
        rawConfig === null ||
        Array.isArray(rawConfig)) {
        return unknownProperties;
    }
    if (rawDefaults === undefined || rawDefaults == null) {
        // the top level property should have been defined, these could be and
        // probably are custom properties.
        return unknownProperties;
    }
    if (typeof rawDefaults !== "object") {
        throw new TypeError("default and normal config are out of sync");
    }
    const defaultConfig = rawDefaults;
    const config = rawConfig;
    for (const key of Object.keys(config)) {
        if (!(key in defaultConfig)) {
            unknownProperties.push("/" + [...currentPathProperties, key].join("/"));
        }
        else {
            const unknownSubProperties = getUnknownPropertiesHelper(config[key], defaultConfig[key], [...currentPathProperties, key]);
            unknownProperties.push(...unknownSubProperties);
        }
    }
    return unknownProperties;
}
/**
 * Return a list of JSON paths to properties in the given config object that are not present in the default config.
 * This is used to detect typos in the config file.
 */
function getUnknownConfigPropertyPaths(config) {
    if (typeof config !== "object" || config === null) {
        return [];
    }
    return getUnknownPropertiesHelper(config, defaultConfig, []);
}
/**
 * Ensures we have an absolute path for storage
 *
 * @param dataPath The dataPath from either the bot or appservice configs
 * @returns The storagePath used for the databases
 */
function getStoragePath(dataPath) {
    return path_1.default.isAbsolute(dataPath)
        ? dataPath
        : path_1.default.join(__dirname, "../", dataPath);
}
//# sourceMappingURL=config.js.map