"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderActionResult = renderActionResult;
exports.printActionResult = printActionResult;
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
function renderErrorItem(error) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("a", { href: error.room.toPermalink() }, error.room.toRoomIDOrAlias()),
        " -",
        " ",
        error.message));
}
/**
 * Render a message to show to the user after taking an action in a room or a set of rooms.
 * @param client A matrix client.
 * @param errors Any errors associated with the action.
 * @param options.title To give context about what the action was, shown when there are errors.
 * @param options.noErrorsText To show when there are no errors.
 * @param options.skipNoErrors is ineffective and does nothing, it is an option for the accompnying `printActionResult`.
 * @returns A `DocumentNode` fragment that can be sent to Matrix or incorperated into another message.
 */
async function renderActionResult(errors, { title = "There were errors updating protected rooms.", noErrorsText = "Done updating rooms - no errors.", } = {}) {
    if (errors.length === 0) {
        return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("font", { color: "#00cc00" }, noErrorsText)));
    }
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("font", { color: "#ff0000" },
            title,
            interface_manager_1.DeadDocumentJSX.JSXFactory("br", null)),
        interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("font", { color: "#ff0000" },
                    interface_manager_1.DeadDocumentJSX.JSXFactory("b", null,
                        errors.length,
                        " errors updating protected rooms!"),
                    interface_manager_1.DeadDocumentJSX.JSXFactory("br", null))),
            interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, errors.map((error) => renderErrorItem(error))))));
}
/**
 * Render a message to represent the outcome of an action in an update.
 * @param client A matrix client to send a notice with.
 * @param roomID The room to send the notice to.
 * @param errors Any errors that are a result of the action.
 * @param options.title To give context about what the action was, shown when there are errors.
 * @param options.noErrorsText To show when there are no errors.
 * @returns
 */
async function printActionResult(messageSender, roomID, errors, renderOptions = {}) {
    const sendResult = await (0, mps_interface_adaptor_1.sendMatrixEventsFromDeadDocument)(messageSender, roomID, interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, await renderActionResult(errors, renderOptions)), {});
    if ((0, matrix_protection_suite_1.isError)(sendResult)) {
        return sendResult.elaborate("Could not printActionResult to the management room.");
    }
    return (0, matrix_protection_suite_1.Ok)(undefined);
}
//# sourceMappingURL=RoomUpdateError.js.map