"use strict";
// Copyright 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagDynamicEnvironment = exports.TagDynamicEnvironmentEntry = exports.FringeWalker = exports.SimpleFringeRenderer = exports.FringeType = exports.EmptyFragment = exports.NodeTag = void 0;
exports.addChild = addChild;
exports.getChildren = getChildren;
exports.getFirstChild = getFirstChild;
exports.makeDocumentNode = makeDocumentNode;
exports.makeLeafNode = makeLeafNode;
exports.addText = addText;
exports.addInlineCode = addInlineCode;
exports.addPreformattedText = addPreformattedText;
const super_cool_stream_1 = require("@gnuxie/super-cool-stream");
// These are NOT necessarily HTML tags.
var NodeTag;
(function (NodeTag) {
    NodeTag["TextNode"] = "text";
    NodeTag["InlineCode"] = "code";
    NodeTag["PreformattedText"] = "pre";
    NodeTag["Root"] = "root";
    NodeTag["Strong"] = "strong";
    NodeTag["Emphasis"] = "em";
    NodeTag["Paragraph"] = "p";
    NodeTag["HeadingOne"] = "h1";
    NodeTag["HeadingTwo"] = "h2";
    NodeTag["HeadingThree"] = "h3";
    NodeTag["HeadingFour"] = "h4";
    NodeTag["HeadingFive"] = "h5";
    NodeTag["HeadingSix"] = "h6";
    NodeTag["UnorderedList"] = "ul";
    NodeTag["OrderedList"] = "ol";
    NodeTag["ListItem"] = "li";
    NodeTag["LineBreak"] = "br";
    NodeTag["BoldFace"] = "b";
    NodeTag["ItalicFace"] = "i";
    NodeTag["Anchor"] = "a";
    NodeTag["Fragment"] = "fragment";
    NodeTag["Details"] = "details";
    NodeTag["Summary"] = "summary";
    NodeTag["Font"] = "font";
    NodeTag["Span"] = "span";
    NodeTag["HorizontalRule"] = "hr";
})(NodeTag || (exports.NodeTag = NodeTag = {}));
exports.EmptyFragment = makeDocumentNode(NodeTag.Fragment);
function addChild(node) {
    if (this.children.includes(node)) {
        return node;
    }
    this.children.push(node);
    return node;
}
function getChildren() {
    return [...this.children];
}
function getFirstChild() {
    return this.children.at(0);
}
function makeDocumentNode(tag, parent = null) {
    const node = {
        tag,
        leafNode: false,
        parent,
        children: [],
        attributeMap: new Map(),
        addChild,
        getChildren,
        getFirstChild,
    };
    return node;
}
function makeLeafNode(tag, parent, data) {
    const leaf = { tag, parent, data, leafNode: true };
    parent.addChild(leaf);
    return leaf;
}
// lol no mixins so addChild  can't be protected on DocumentNode
// it's a yoke.
function addText(data) {
    return this.addChild(makeLeafNode(NodeTag.TextNode, this, data));
}
function addInlineCode(data) {
    return this.addChild(makeLeafNode(NodeTag.InlineCode, this, data));
}
function addPreformattedText(data) {
    return this.addChild(makeLeafNode(NodeTag.PreformattedText, this, data));
}
/**
 * We use a Fringe to render.
 */
var FringeType;
(function (FringeType) {
    FringeType["Pre"] = "pre";
    FringeType["Leaf"] = "leaf";
    FringeType["Post"] = "post";
})(FringeType || (exports.FringeType = FringeType = {}));
// https://dl.acm.org/doi/pdf/10.1145/165507.165514
// If performance becomes a concern then a generator should be written
// but for rendering incrementally i think constructing the fringe is acceptable.
// So long as no one uses `Flat` directly and there's some inversion of control
// then we'll be fine.
function fringeInternalNode(node, flat) {
    if (node.getChildren().length === 0) {
        return flat;
    }
    else {
        return node
            .getChildren()
            .reduce((previous, child) => {
            return fringe(child, previous);
        }, flat);
    }
}
function fringe(node, flat = []) {
    if (node.leafNode) {
        flat.push({ type: FringeType.Leaf, node });
        return flat;
    }
    else {
        flat.push({ type: FringeType.Pre, node });
        flat = fringeInternalNode(node, flat);
        flat.push({ type: FringeType.Post, node });
        return flat;
    }
}
class SimpleFringeRenderer {
    constructor() {
        this.preRenderers = new Map();
        this.leafRenderers = new Map();
        this.postRenderers = new Map();
    }
    getRenderer(table, type, tag) {
        const entry = table.get(tag);
        if (entry) {
            return entry;
        }
        throw new TypeError(`Couldn't find a ${type} renderer for ${tag}`);
    }
    getPreRenderer(tag) {
        return this.getRenderer(this.preRenderers, FringeType.Pre, tag);
    }
    getLeafRenderer(tag) {
        return this.getRenderer(this.leafRenderers, FringeType.Leaf, tag);
    }
    getPostRenderer(tag) {
        return this.getRenderer(this.postRenderers, FringeType.Post, tag);
    }
    internRenderer(type, tag, table, renderer) {
        if (table.has(tag)) {
            throw new TypeError(`There is already a renderer registered for ${type} ${tag}`);
        }
        table.set(tag, renderer);
    }
    registerRenderer(type, tag, renderer) {
        // The casting in here is evil. Not sure how to fix it.
        switch (type) {
            case FringeType.Pre:
                this.internRenderer(type, tag, this.preRenderers, renderer);
                break;
            case FringeType.Leaf:
                this.internRenderer(type, tag, this.leafRenderers, renderer);
                break;
            case FringeType.Post:
                this.internRenderer(type, tag, this.postRenderers, renderer);
                break;
        }
        return this;
    }
    registerInnerNode(tag, pre, post) {
        this.internRenderer(FringeType.Pre, tag, this.preRenderers, pre);
        this.internRenderer(FringeType.Post, tag, this.postRenderers, post);
        return this;
    }
}
exports.SimpleFringeRenderer = SimpleFringeRenderer;
const COMMITTABLE_NODES = new Set([
    NodeTag.HeadingOne,
    NodeTag.ListItem,
    NodeTag.Paragraph,
    NodeTag.PreformattedText,
    NodeTag.UnorderedList,
    NodeTag.Root,
]);
class FringeStream extends super_cool_stream_1.StandardSuperCoolStream {
}
/**
 * The FringeWalker allows for the implementation of an incremental
 * renderer.
 * Each increment is defined by the fist leaf node to be rendered
 * or the first inner node to have all of its leaves renderered.
 * @param Context is a static context that should be provided to each render function.
 */
class FringeWalker {
    constructor(root, context, renderer, commitHook) {
        this.root = root;
        this.context = context;
        this.renderer = renderer;
        this.commitHook = commitHook;
        this.dynamicEnvironment = new TagDynamicEnvironment();
        this.stream = new FringeStream(fringe(root));
    }
    increment() {
        const renderInnerNode = (node) => {
            if (node.node.leafNode) {
                throw new TypeError("Leaf nodes should not be in the Pre/Post position");
            }
            const renderer = node.type === FringeType.Pre
                ? this.renderer.getPreRenderer(node.node.tag)
                : this.renderer.getPostRenderer(node.node.tag);
            renderer(node.type, node.node, this.context, this.dynamicEnvironment);
            return node.node;
        };
        const postNode = (node) => {
            if (node.node.leafNode) {
                throw new TypeError("Leaf nodes should not be in the Pre/Post position");
            }
            renderInnerNode(node);
            this.dynamicEnvironment.pop(node.node);
            return node.node;
        };
        const isAnnotatedNodeCommittable = (node) => {
            return (COMMITTABLE_NODES.has(node.node.tag) && node.type === FringeType.Post);
        };
        while (this.stream.peekItem() &&
            !isAnnotatedNodeCommittable(this.stream.peekItem())) {
            const annotatedNode = this.stream.readItem();
            if (annotatedNode === undefined) {
                throw new TypeError(`Stream code is wrong`);
            }
            switch (annotatedNode.type) {
                case FringeType.Pre:
                    renderInnerNode(annotatedNode);
                    break;
                case FringeType.Post:
                    postNode(annotatedNode);
                    break;
                case FringeType.Leaf:
                    if (!annotatedNode.node.leafNode) {
                        throw new TypeError("Leaf nodes should not be marked as an inner node");
                    }
                    this.renderer.getLeafRenderer(annotatedNode.node.tag)(annotatedNode.node.tag, annotatedNode.node, this.context);
                    break;
                default:
                    throw new TypeError(`Uknown fringe type ${annotatedNode.type}`);
            }
        }
        if (this.stream.peekItem() === undefined) {
            return undefined;
        }
        const documentNode = postNode(this.stream.readItem());
        this.commitHook(documentNode, this.context);
        return documentNode;
    }
}
exports.FringeWalker = FringeWalker;
class TagDynamicEnvironmentEntry {
    constructor(node, value, previous) {
        this.node = node;
        this.value = value;
        this.previous = previous;
    }
}
exports.TagDynamicEnvironmentEntry = TagDynamicEnvironmentEntry;
/**
 * A dynamic environment is just an environment of bindings that is made
 * by shadowing previous bindings and pushing and popping bindings "dynamically"
 * for a given variable with some thing.
 *
 * In this example, we push and pop bindings with `DocumentNode`s.
 * For example, if you make a binding to a variable called `indentationLevel`
 * to set it to `1` from a `<ul>` node, then this binding should be popped
 * when the `FringeWalker` reaches the post node (`</ul>`).
 * Howerver, if we encounter another `<ul>`, we can read the existing value
 * for `indentationLevel`, increment it and create a new binding
 * that shadows the existing one. This too will get popped once we encounter
 * the post node for this `<ul>` node (`</ul>`).
 *
 * This makes it very easy to express a situation where you modify and
 * restore variables that depend on node depth when walking the fringe,
 * as the restoration of previous values can be handled automatically for us.
 */
class TagDynamicEnvironment {
    constructor() {
        this.environments = new Map();
    }
    read(variableName) {
        const variableEntry = this.environments.get(variableName);
        if (variableEntry) {
            return variableEntry.value;
        }
        else {
            throw new TypeError(`The variable ${variableName} is unbound.`);
        }
    }
    write(variableName, value) {
        const variableEntry = this.environments.get(variableName);
        if (variableEntry) {
            return (variableEntry.value = value);
        }
        else {
            throw new TypeError(`The variable ${variableName} is unbound.`);
        }
    }
    bind(variableName, node, value) {
        const entry = this.environments.get(variableName);
        const newEntry = new TagDynamicEnvironmentEntry(node, value, entry);
        this.environments.set(variableName, newEntry);
        return value;
    }
    pop(node) {
        for (const [variableName, environment] of this.environments.entries()) {
            if (Object.is(environment === null || environment === void 0 ? void 0 : environment.node, node)) {
                this.environments.set(variableName, environment === null || environment === void 0 ? void 0 : environment.previous);
            }
        }
    }
}
exports.TagDynamicEnvironment = TagDynamicEnvironment;
//# sourceMappingURL=DeadDocument.js.map