"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardMatrixInterfaceAdaptor = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const Command_1 = require("../Command");
const Adaptor_1 = require("../Adaptor");
const DefaultRenderers_1 = require("./DefaultRenderers");
class StandardMatrixInterfaceAdaptor {
    constructor(adaptorToCommandContextTranslator, invocationInformationFromEventContext, interfaceAdaptorCallbacks, commandInvokerCallbacks) {
        this.adaptorToCommandContextTranslator = adaptorToCommandContextTranslator;
        this.invocationInformationFromEventContext = invocationInformationFromEventContext;
        this.renderers = new Map();
        this.commandInvoker = new Adaptor_1.StandardCommandInvoker(commandInvokerCallbacks);
        this.callbacks = interfaceAdaptorCallbacks;
    }
    async invoke(command, adaptorContext, matrixEventContext) {
        const renderer = this.findRendererForCommandDescription(command.description);
        const commandContext = this.adaptorToCommandContextTranslator.translateContext(command.description, adaptorContext);
        const commandResult = await this.commandInvoker.invoke(commandContext, this.invocationInformationFromEventContext(matrixEventContext), command);
        return (await this.runRenderersOnCommandResult(command, commandResult, renderer, adaptorContext, matrixEventContext));
    }
    findRendererForCommandDescription(commandDescription) {
        const renderer = this.renderers.get(commandDescription);
        if (renderer === undefined) {
            throw new TypeError(`There is no renderer defined for the command ${commandDescription.summary}`);
        }
        return renderer;
    }
    async runRenderersOnCommandResult(command, commandResult, renderer, adaptorContext, matrixEventContext) {
        const renderResults = await Promise.all([
            this.maybeRunDefaultRenderer(renderer, adaptorContext, matrixEventContext, command, commandResult),
            this.maybeRunJSXRenderer(renderer, adaptorContext, matrixEventContext, command, commandResult),
            this.maybeRunArbritraryRenderer(renderer, adaptorContext, matrixEventContext, commandResult),
        ]);
        for (const result of renderResults) {
            if ((0, typescript_result_1.isError)(result)) {
                this.callbacks.rendererFailedCB(adaptorContext, matrixEventContext, command, result.error);
            }
        }
        return commandResult;
    }
    async maybeRunDefaultRenderer(renderer, adaptorContext, eventContext, command, commandResult) {
        if (!renderer.isAlwaysSupposedToUseDefaultRenderer) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        return await this.callbacks.defaultRenderer(adaptorContext, eventContext, command, commandResult);
    }
    async maybeRunJSXRenderer(renderer, adaptorContext, eventContext, command, commandResult) {
        if (command.description.parametersDescription.keywords.keywordDescriptions["no-confirm"] !== undefined &&
            !command.isPartial &&
            !command.keywords.getKeywordValue("no-confirm", false)) {
            const finalDocument = renderer.confirmationPromptJSXRenderer
                ? renderer.confirmationPromptJSXRenderer(commandResult)
                : (0, DefaultRenderers_1.renderConfirmationPrompt)(commandResult);
            if ((0, typescript_result_1.isError)(finalDocument)) {
                return finalDocument;
            }
            if (finalDocument.ok === undefined) {
                return (0, typescript_result_1.Ok)(undefined); // Renderer is telling us it doesn't want to render anything.
            }
            return await this.callbacks.matrixEventsFromConfirmationPrompt(adaptorContext, eventContext, command, finalDocument.ok);
        }
        if (!renderer.JSXRenderer) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        const document = renderer.JSXRenderer(commandResult);
        if ((0, typescript_result_1.isError)(document)) {
            return document;
        }
        if (document.ok === undefined) {
            return (0, typescript_result_1.Ok)(undefined); // Renderer is telling us it doesn't want to render anything.
        }
        return await this.callbacks.matrixEventsFromDeadDocument(adaptorContext, eventContext, document.ok);
    }
    async maybeRunArbritraryRenderer(renderer, adaptorContext, eventContext, commandResult) {
        if (renderer.arbritraryRenderer) {
            return await renderer.arbritraryRenderer(adaptorContext, eventContext, commandResult);
        }
        else {
            return (0, typescript_result_1.Ok)(undefined);
        }
    }
    registerRendererDescription(commandDescription, rendererDescription) {
        this.renderers.set(commandDescription, rendererDescription);
        return this;
    }
    describeRenderer(commandDescription, rendererDescription) {
        var _a;
        return this.registerRendererDescription(commandDescription, {
            ...rendererDescription,
            isAlwaysSupposedToUseDefaultRenderer: (_a = rendererDescription.isAlwaysSupposedToUseDefaultRenderer) !== null && _a !== void 0 ? _a : true,
        });
    }
    async parseAndInvoke(partialCommand, adaptorContext, eventContext) {
        const renderer = this.findRendererForCommandDescription(partialCommand.description);
        const commandArguments = partialCommand.stream.rest();
        const parseResult = this.commandInvoker.parseCommand(this.invocationInformationFromEventContext(eventContext), partialCommand);
        if ((0, typescript_result_1.isError)(parseResult)) {
            if (parseResult.error instanceof Command_1.PromptRequiredError) {
                const parameter = parseResult.error.parameterRequiringPrompt;
                if (parameter.prompt === undefined) {
                    throw new TypeError(`A PromptRequiredError was given for a parameter which doesn't support prompts, this shouldn't happen`);
                }
                const commandContext = this.adaptorToCommandContextTranslator.translateContext(partialCommand.description, adaptorContext);
                const promptOptionsResult = await parameter.prompt(commandContext // weh, we know we have the right adaptorContext, it's just being annoying while we avoid controvariance.
                );
                if ((0, typescript_result_1.isError)(promptOptionsResult)) {
                    return promptOptionsResult.elaborate(`Failed to get prompt options for ${parameter.name} while parsing the command "${partialCommand.designator.join(" ")}".`);
                }
                const promptOptions = promptOptionsResult.ok;
                const promptResult = promptOptions.default === undefined
                    ? await this.callbacks.promptSuggestions(adaptorContext, eventContext, parameter, partialCommand, promptOptions.suggestions, commandArguments)
                    : await this.callbacks.promptDefault(adaptorContext, eventContext, parameter, partialCommand, promptOptions.default, commandArguments);
                if ((0, typescript_result_1.isError)(promptResult)) {
                    return promptResult.elaborate(`Failed to prompt the user for ${parameter.name} while parsing the command "${partialCommand.designator.join(" ")}".`);
                }
                else {
                    return (0, typescript_result_1.Ok)(undefined);
                }
            }
            else {
                return (await this.runRenderersOnCommandResult(partialCommand, parseResult, renderer, adaptorContext, eventContext));
            }
        }
        return await this.invoke(parseResult.ok, adaptorContext, eventContext);
    }
    isDescribingRendererForCommand(commandDescription) {
        return this.renderers.has(commandDescription);
    }
    renderedCommands() {
        return [...this.renderers.keys()];
    }
}
exports.StandardMatrixInterfaceAdaptor = StandardMatrixInterfaceAdaptor;
//# sourceMappingURL=MatrixInterfaceAdaptor.js.map