import { Result, ResultError } from '@gnuxie/typescript-result';
import { ConfigDescription } from './ConfigDescription';
import { ConfigParseError } from './ConfigParseError';
import { StaticEncode, TObject } from '@sinclair/typebox';
import { EDStatic } from '../Interface/Static';
export type ConfigRecoveryOption = {
    readonly description: string;
    recover(): Promise<Result<void>>;
};
/**
 * Allows the client to load persistent config data.
 * The schema gets verified before its returned.
 * If there is an error with the data, recovery options are provided.
 *
 * Draupnir maintains a list of these which are editable generically
 * via safe mode.
 */
export interface PersistentConfigData<T extends TObject = TObject> {
    readonly description: ConfigDescription<T>;
    requestParsedConfig(): Promise<Result<EDStatic<T> | undefined, ResultError | ConfigParseError>>;
    saveConfig(config: EDStatic<T>): Promise<Result<void>>;
    makeRecoveryOptionsForProperty(config: Record<string, unknown>, key: string): ConfigRecoveryOption[];
    makeRecoveryOptionsForPropertyItem(config: Record<string, unknown[]>, key: string, index: number): ConfigRecoveryOption[];
    makeRecoveryOptionsForError(config: Record<string, unknown>, error: ResultError): ConfigRecoveryOption[];
    addRecoveryOptionsToResult<T = unknown>(config: Record<string, unknown>, result: Result<T>): Result<T>;
    reportUseError(message: string, options: {
        path: string;
        value: unknown;
        cause: ResultError;
    }): Promise<Result<never>>;
}
/**
 * The backend for the `PersistentConfigData` class.
 * Does not serialize or decode data in any way beyond basic JSON deserialization.
 * This is so that the `PersistentConfigData` class has full control over serialization and transformation.
 * @typeParam TEncodedShape This is the shape of the data once it has been transformed into a plain JSON object,
 * without any other JS objects or types.
 */
export interface PersistentConfigBackend<TEncodedShape extends Record<string, unknown> = Record<string, unknown>> {
    requestUnparsedConfig(): Promise<Result<Record<string, unknown> | undefined>>;
    saveEncodedConfig(data: TEncodedShape): Promise<Result<void>>;
}
export declare class StandardPersistentConfigData<TConfigSchema extends TObject> implements PersistentConfigData<TConfigSchema> {
    readonly description: ConfigDescription<TConfigSchema>;
    private readonly backend;
    constructor(description: ConfigDescription<TConfigSchema>, backend: PersistentConfigBackend<StaticEncode<TConfigSchema>>);
    requestParsedConfig(): Promise<Result<EDStatic<TConfigSchema> | undefined, ResultError | ConfigParseError>>;
    saveConfig(config: EDStatic<TConfigSchema>): Promise<Result<void>>;
    private makeRecoveryOptionForConfig;
    makeRecoveryOptionsForProperty(config: Record<string, unknown>, key: string): ConfigRecoveryOption[];
    makeRecoveryOptionsForPropertyItem(config: Record<string, unknown[]>, key: string, index: number): ConfigRecoveryOption[];
    makeRecoveryOptionsForError(config: Record<string, unknown>, error: ResultError): ConfigRecoveryOption[];
    addRecoveryOptionsToResult<T = unknown>(config: Record<string, unknown>, result: Result<T>): Result<T>;
    reportUseError(message: string, options: {
        path: string;
        value: unknown;
        cause: ResultError;
    }): Promise<Result<never>>;
}
//# sourceMappingURL=PersistentConfigData.d.ts.map