"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardLifetime = void 0;
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
const typescript_result_1 = require("@gnuxie/typescript-result");
const Logger_1 = require("../Logging/Logger");
const log = new Logger_1.Logger('Lifetime');
async function callDisposeHandle(handle) {
    if (typeof handle === 'function') {
        await handle();
    }
    else if (Symbol.dispose in handle) {
        handle[Symbol.dispose]();
    }
    else {
        await handle[Symbol.asyncDispose]();
    }
}
class StandardLifetime {
    constructor(options = {}) {
        var _a;
        this.controller = new AbortController();
        this.callbacks = new Set();
        this.resolveDisposed = undefined;
        this.parent = options.parent;
        (_a = this.parent) === null || _a === void 0 ? void 0 : _a.onDispose(this);
        this.disposedPromise = new Promise((resolve) => {
            this.resolveDisposed = resolve;
        });
    }
    isInDisposal() {
        return this.controller.signal.aborted;
    }
    onDispose(callback) {
        if (this.isInDisposal()) {
            throw new TypeError('You are registering a resource with the Lifetime non atomically. You must only register resources immediately and atomically with resource allocation. Use the allocateResource method.');
        }
        else {
            this.callbacks.add(callback);
            return this;
        }
    }
    forget(callback) {
        // we don't want to delete something we are in the process of disposing.
        if (this.isInDisposal()) {
            return this;
        }
        this.callbacks.delete(callback);
        return this;
    }
    async forgetAndDispose(callback) {
        if (this.isInDisposal()) {
            await this.disposedPromise;
            return;
        }
        this.forget(callback);
        await callDisposeHandle(callback);
    }
    toAbortSignal() {
        return this.controller.signal;
    }
    async [Symbol.asyncDispose]() {
        var _a;
        if (this.isInDisposal()) {
            return this.disposedPromise;
        }
        this.controller.abort();
        for (const callback of this.callbacks) {
            try {
                await callDisposeHandle(callback);
            }
            catch (error) {
                log.error('Error during disposal callback', error);
            }
        }
        this.callbacks.clear();
        (_a = this.parent) === null || _a === void 0 ? void 0 : _a.forget(this);
        if (this.resolveDisposed === undefined) {
            // It's possible that dispose has been called when there are no callbacks.
            // If we process this in the next tick the resolver should have been
            // assigned by that point.
            return new Promise((resolve, reject) => {
                if (this.resolveDisposed === undefined) {
                    reject(new TypeError('resolveDisposed is undefined during disposal. This should not be possible.'));
                    return;
                }
                else {
                    this.resolveDisposed();
                    resolve();
                }
            });
        }
        this.resolveDisposed();
    }
    toChild() {
        return this.allocateResource(() => (0, typescript_result_1.Ok)(new StandardLifetime({ parent: this })), (child) => child);
    }
    allocateResource(factory, disposer) {
        if (this.isInDisposal()) {
            return typescript_result_1.ResultError.Result('Resource was not initialized: Lifetime is in disposal. Use isInDisposal to check before using this method.');
        }
        const resource = factory(this);
        if ((0, typescript_result_1.isError)(resource)) {
            return resource;
        }
        this.onDispose(disposer(resource.ok));
        return resource;
    }
    allocateDisposable(factory) {
        return this.allocateResource(factory, (resource) => resource);
    }
    async withDisposalBlocked(cb) {
        if (this.isInDisposal()) {
            return typescript_result_1.ResultError.Result('Lifetime is in disposal, so disposal cannot be blocked');
        }
        const blockingPromise = cb();
        const handle = () => blockingPromise.then(() => { });
        this.onDispose(handle);
        try {
            return await blockingPromise;
        }
        finally {
            this.forget(handle);
        }
    }
    async allocateResourceAsync(factory, disposer) {
        return await this.withDisposalBlocked(async () => {
            const resource = await factory(this);
            if ((0, typescript_result_1.isError)(resource)) {
                return resource;
            }
            if (this.isInDisposal()) {
                await callDisposeHandle(disposer(resource.ok));
                return typescript_result_1.ResultError.Result('Resource had to be disposed after allocation because Lifetime entered disposal');
            }
            else {
                this.onDispose(disposer(resource.ok));
                return resource;
            }
        });
    }
    async allocateDisposableAsync(factory) {
        return await this.allocateResourceAsync(factory, (resource) => resource);
    }
}
exports.StandardLifetime = StandardLifetime;
//# sourceMappingURL=Lifetime.js.map