"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyRuleMatchType = exports.Recommendation = void 0;
exports.normaliseRecommendation = normaliseRecommendation;
exports.makeReversedHashedPolicy = makeReversedHashedPolicy;
exports.parsePolicyRule = parsePolicyRule;
// Copyright 2022 - 2023 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const PolicyEvents_1 = require("../MatrixTypes/PolicyEvents");
const typescript_result_1 = require("@gnuxie/typescript-result");
var Recommendation;
(function (Recommendation) {
    /// The rule recommends a "ban".
    ///
    /// The actual semantics for this "ban" may vary, e.g. room ban,
    /// server ban, ignore user, etc. To determine the semantics for
    /// this "ban", clients need to take into account the context for
    /// the list, e.g. how the rule was imported.
    Recommendation["Ban"] = "m.ban";
    /**
     * This is a rule that recommends allowing a user to participate.
     * Used for the construction of allow lists.
     */
    Recommendation["Allow"] = "org.matrix.mjolnir.allow";
    /**
     * This recommendation is to takedown the entity and is usually reserved
     * for content that needs to be removed asap.
     */
    Recommendation["Takedown"] = "org.matrix.msc4204.takedown";
    Recommendation["Unknown"] = "unknown";
})(Recommendation || (exports.Recommendation = Recommendation = {}));
/**
 * All variants of recommendation `m.ban`
 */
const RECOMMENDATION_BAN_VARIANTS = [
    // Stable
    Recommendation.Ban,
    // Unstable prefix, for compatibility.
    'org.matrix.mjolnir.ban',
];
const RECOMMENDATION_ALLOW_VARIANTS = [
    // Unstable
    Recommendation.Allow,
];
const RECOMMENDATION_TAKEDOWN_VARIANTS = [
    Recommendation.Takedown,
    'm.takedown',
];
function normaliseRecommendation(recommendation) {
    if (RECOMMENDATION_BAN_VARIANTS.includes(recommendation)) {
        return Recommendation.Ban;
    }
    else if (RECOMMENDATION_ALLOW_VARIANTS.includes(recommendation)) {
        return Recommendation.Allow;
    }
    else if (RECOMMENDATION_TAKEDOWN_VARIANTS.includes(recommendation)) {
        return Recommendation.Takedown;
    }
    else {
        return Recommendation.Unknown;
    }
}
function makeReversedHashedPolicy(entity, hashedPolicy) {
    var _a;
    return Object.freeze({
        entity,
        kind: hashedPolicy.kind,
        recommendation: hashedPolicy.recommendation,
        sourceEvent: hashedPolicy.sourceEvent,
        reason: (_a = hashedPolicy.reason) !== null && _a !== void 0 ? _a : '<no reason supplied>',
        matchType: PolicyRuleMatchType.Literal,
        isMatch(entity) {
            return this.entity === entity;
        },
        isReversedFromHashedPolicy: true,
    });
}
function parsePolicyRule(event) {
    var _a, _b;
    if (!('entity' in event.content)) {
        const hashes = 
        // we need the expressions mare:
        // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
        ('hashes' in event.content && event.content.hashes) ||
            ('org.matrix.msc4205.hashes' in event.content &&
                event.content['org.matrix.msc4205.hashes']);
        if (!hashes) {
            return typescript_result_1.ResultError.Result('There is a missing entity in the policy rule');
        }
        return (0, typescript_result_1.Ok)(Object.freeze({
            recommendation: normaliseRecommendation(event.content.recommendation),
            kind: (0, PolicyEvents_1.normalisePolicyRuleType)(event.type),
            hashes,
            sourceEvent: event,
            matchType: PolicyRuleMatchType.HashedLiteral,
            ...(event.content.reason ? { reason: event.content.reason } : {}),
        }));
    }
    if (/[*?]/.test(event.content.entity)) {
        return (0, typescript_result_1.Ok)(Object.freeze({
            glob: new matrix_basic_types_1.MatrixGlob(event.content.entity),
            entity: event.content.entity,
            recommendation: normaliseRecommendation(event.content.recommendation),
            kind: (0, PolicyEvents_1.normalisePolicyRuleType)(event.type),
            sourceEvent: event,
            matchType: PolicyRuleMatchType.Glob,
            reason: (_a = event.content.reason) !== null && _a !== void 0 ? _a : '<no reason supplied>',
            isMatch(entity) {
                return this.glob.test(entity);
            },
        }));
    }
    else {
        return (0, typescript_result_1.Ok)(Object.freeze({
            entity: event.content.entity,
            recommendation: normaliseRecommendation(event.content.recommendation),
            kind: (0, PolicyEvents_1.normalisePolicyRuleType)(event.type),
            sourceEvent: event,
            matchType: PolicyRuleMatchType.Literal,
            reason: (_b = event.content.reason) !== null && _b !== void 0 ? _b : '<no reason supplied>',
            isMatch(entity) {
                return this.entity === entity;
            },
        }));
    }
}
var PolicyRuleMatchType;
(function (PolicyRuleMatchType) {
    PolicyRuleMatchType["Literal"] = "literal";
    PolicyRuleMatchType["Glob"] = "glob";
    PolicyRuleMatchType["HashedLiteral"] = "hashed-literal";
})(PolicyRuleMatchType || (exports.PolicyRuleMatchType = PolicyRuleMatchType = {}));
//# sourceMappingURL=PolicyRule.js.map