"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Capability = void 0;
exports.registerCapabilityProvider = registerCapabilityProvider;
exports.findCapabilityProvider = findCapabilityProvider;
exports.describeCapabilityProvider = describeCapabilityProvider;
exports.findCapabilityProviderSet = findCapabilityProviderSet;
exports.findCompatibleCapabilityProviders = findCompatibleCapabilityProviders;
exports.findSimulatedCapabilityProvider = findSimulatedCapabilityProvider;
// Copyright 2023 - 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
const typebox_1 = require("@sinclair/typebox");
const CapabilityInterface_1 = require("./CapabilityInterface");
exports.Capability = typebox_1.Type.Object({
    requiredPermissions: typebox_1.Type.Array(typebox_1.Type.String()),
    requiredEventPermissions: typebox_1.Type.Array(typebox_1.Type.String()),
    requiredStatePermissions: typebox_1.Type.Array(typebox_1.Type.String()),
});
const PROVIDER_DESCRIPTIONS = new Map();
const PROVIDER_DESCRIPTIONS_FOR_INTERFACE = new Map();
const SIMULATED_CAPABILITY_PROVIDERS = new Map();
function registerCapabilityProvider(description) {
    var _a;
    if (PROVIDER_DESCRIPTIONS.has(description.name)) {
        throw new TypeError(`There is already a consequence provider named ${description.name}`);
    }
    if (description.isSimulated) {
        if (SIMULATED_CAPABILITY_PROVIDERS.has(description.interface.name)) {
            throw new TypeError(`There is already a simualted capability provider for the ${description.interface.name} interface`);
        }
        SIMULATED_CAPABILITY_PROVIDERS.set(description.interface.name, description);
    }
    PROVIDER_DESCRIPTIONS.set(description.name, description);
    PROVIDER_DESCRIPTIONS_FOR_INTERFACE.set(description.interface.name, [
        ...((_a = PROVIDER_DESCRIPTIONS_FOR_INTERFACE.get(description.interface.name)) !== null && _a !== void 0 ? _a : []),
        description,
    ]);
}
function findCapabilityProvider(name) {
    return PROVIDER_DESCRIPTIONS.get(name);
}
function describeCapabilityProvider({ name, description, interface: interfaceName, isSimulated, factory, }) {
    const entry = (0, CapabilityInterface_1.findCapabilityInterface)(interfaceName);
    if (entry === undefined) {
        throw new TypeError(`Cannot find a CapabilityInterface named ${interfaceName}`);
    }
    registerCapabilityProvider({
        name,
        description,
        interface: entry,
        ...(isSimulated ? { isSimulated } : {}),
        factory,
    });
}
function findCapabilityProviderSet(names) {
    const set = {};
    for (const [key, name] of Object.entries(names)) {
        const capabilityProvider = findCapabilityProvider(name);
        if (capabilityProvider === undefined) {
            throw new TypeError(`Couldn't find a capability provider named ${name}`);
        }
        Object.assign(set, { [key]: capabilityProvider });
    }
    return set;
}
function findCompatibleCapabilityProviders(interfaceName) {
    var _a;
    return (_a = PROVIDER_DESCRIPTIONS_FOR_INTERFACE.get(interfaceName)) !== null && _a !== void 0 ? _a : [];
}
function findSimulatedCapabilityProvider(interfaceName) {
    return SIMULATED_CAPABILITY_PROVIDERS.get(interfaceName);
}
//# sourceMappingURL=CapabilityProvider.js.map