"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MjolnirPolicyRoomsConfig = void 0;
// Copyright (C) 2023 - 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const await_lock_1 = __importDefault(require("await-lock"));
const PolicyListConfig_1 = require("./PolicyListConfig");
const Logger_1 = require("../../Logging/Logger");
const PersistentConfigData_1 = require("../../Config/PersistentConfigData");
const MjolnirPolicyRoomsDescription_1 = require("./MjolnirPolicyRoomsDescription");
const immutable_1 = require("immutable");
const typescript_result_1 = require("@gnuxie/typescript-result");
const log = new Logger_1.Logger('MjolnirPolicyRoomsConfig');
class MjolnirPolicyRoomsConfig {
    constructor(config, roomJoiner, watchedLists) {
        this.config = config;
        this.roomJoiner = roomJoiner;
        this.watchedLists = watchedLists;
        this.writeLock = new await_lock_1.default();
        // nothing to do
    }
    static async createFromStore(store, roomJoiner) {
        var _a;
        const config = new PersistentConfigData_1.StandardPersistentConfigData(MjolnirPolicyRoomsDescription_1.MjolnirPolicyRoomsDescription, store);
        const dataResult = await config.requestParsedConfig();
        if ((0, typescript_result_1.isError)(dataResult)) {
            return dataResult.elaborate('Failed to load MjolnirPolicyRoomsConfig from account data');
        }
        const data = (_a = dataResult.ok) !== null && _a !== void 0 ? _a : config.description.getDefaultConfig();
        let watchedLists = (0, immutable_1.Map)();
        for (const [i, reference] of data.references.entries()) {
            const joinResult = await roomJoiner.joinRoom(reference);
            if ((0, typescript_result_1.isError)(joinResult)) {
                log.info(`MjolnirPolicyRoomsConfig:`, data);
                return await config.reportUseError('Unable to join policy room from a provided reference', {
                    path: `/references/${i}`,
                    value: reference,
                    cause: joinResult.error,
                });
            }
            watchedLists = watchedLists.set(joinResult.ok.toRoomIDOrAlias(), joinResult.ok);
        }
        return (0, typescript_result_1.Ok)(new MjolnirPolicyRoomsConfig(config, roomJoiner, watchedLists));
    }
    get allWatchedLists() {
        return [...this.watchedLists.values()].map((room) => ({
            room,
            propagation: PolicyListConfig_1.PropagationType.Direct,
        }));
    }
    async watchList(propagation, list, _options) {
        // More variants could be added under our feet as code changes:
        // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
        if (propagation !== PolicyListConfig_1.PropagationType.Direct) {
            return typescript_result_1.ResultError.Result(`The MjolnirProtectedRoomsConfig does not support watching a list ${list.toPermalink()} with propagation type ${propagation}.`);
        }
        const joinResult = await this.roomJoiner.joinRoom(list);
        if ((0, typescript_result_1.isError)(joinResult)) {
            return joinResult.elaborate(`Could not join the policy room in order to begin watching it.`);
        }
        await this.writeLock.acquireAsync();
        try {
            const storeUpdateResult = await this.config.saveConfig({
                references: [
                    ...this.watchedLists.set(list.toRoomIDOrAlias(), list).values(),
                ],
            });
            if ((0, typescript_result_1.isError)(storeUpdateResult)) {
                return storeUpdateResult;
            }
            this.watchedLists = this.watchedLists.set(list.toRoomIDOrAlias(), list);
        }
        finally {
            this.writeLock.release();
        }
        return (0, typescript_result_1.Ok)(undefined);
    }
    async unwatchList(propagation, list) {
        // More variants could be added under our feet as code changes:
        // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
        if (propagation !== PolicyListConfig_1.PropagationType.Direct) {
            return typescript_result_1.ResultError.Result(`The MjolnirProtectedRoomsConfigUnable does not support watching a list ${list.toPermalink()} with propagation type ${propagation}.`);
        }
        await this.writeLock.acquireAsync();
        try {
            const storeUpdateResult = await this.config.saveConfig({
                references: [
                    ...this.watchedLists.delete(list.toRoomIDOrAlias()).values(),
                ],
            });
            if ((0, typescript_result_1.isError)(storeUpdateResult)) {
                return storeUpdateResult;
            }
            this.watchedLists = this.watchedLists.delete(list.toRoomIDOrAlias());
        }
        finally {
            this.writeLock.release();
        }
        return (0, typescript_result_1.Ok)(undefined);
    }
}
exports.MjolnirPolicyRoomsConfig = MjolnirPolicyRoomsConfig;
//# sourceMappingURL=MjolnirPolicyRoomsConfig.js.map