"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardSetRoomState = void 0;
// Copyright (C) 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const events_1 = __importDefault(require("events"));
const SetRoomState_1 = require("./SetRoomState");
const Action_1 = require("../Interface/Action");
/**
 * Provides immediate (synchronous) access to the room state within a set of rooms.
 * Needs to be backed up with the ProtectedRoomManager which will add and remove rooms
 * via the `SetRoomStateMirror` as they are added and removed from the protected
 * rooms set.
 */
class StandardSetRoomState extends events_1.default {
    constructor() {
        super();
        this.issuers = new Map();
        this.revisionListener = this.stateRevision.bind(this);
    }
    getRevision(room) {
        var _a;
        return (_a = this.issuers.get(room)) === null || _a === void 0 ? void 0 : _a.currentRevision;
    }
    static async create(roomStateManager, roomSet) {
        const setRoomState = new StandardSetRoomState();
        const issuersResult = await Promise.all(roomSet.map((room) => roomStateManager.getRoomStateRevisionIssuer(room)));
        for (const result of issuersResult) {
            if ((0, Action_1.isError)(result)) {
                return result.elaborate(`Unable to fetch a room state revision issuer while creating the SetRoomState`);
            }
            else {
                SetRoomState_1.SetRoomStateMirror.addRoom(setRoomState, result.ok.room, result.ok);
            }
        }
        return (0, Action_1.Ok)(setRoomState);
    }
    static blankSet() {
        return new StandardSetRoomState();
    }
    [SetRoomState_1.SetRoomStateMirrorCord.addRoom](room, roomStateRevisionIssuer) {
        if (this.issuers.has(room.toRoomIDOrAlias())) {
            return;
        }
        this.issuers.set(room.toRoomIDOrAlias(), roomStateRevisionIssuer);
        roomStateRevisionIssuer.on('revision', this.revisionListener);
    }
    [SetRoomState_1.SetRoomStateMirrorCord.removeRoom](room) {
        const issuer = this.issuers.get(room.toRoomIDOrAlias());
        if (issuer === undefined) {
            return;
        }
        this.issuers.delete(room.toRoomIDOrAlias());
        issuer.off('revision', this.revisionListener);
    }
    unregisterListeners() {
        for (const issuer of this.issuers.values()) {
            issuer.off('revision', this.revisionListener);
        }
    }
    get allRooms() {
        return [...this.issuers.values()].map((issuer) => issuer.currentRevision);
    }
    stateRevision(...[nextRevision, changes, previousRevision]) {
        this.emit('revision', nextRevision.room.toRoomIDOrAlias(), nextRevision, changes, previousRevision);
    }
}
exports.StandardSetRoomState = StandardSetRoomState;
//# sourceMappingURL=StandardSetRoomState.js.map