"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirNews = exports.DraupnirNewsProtectionSettings = exports.DraupnirNewsReader = exports.DraupnirNewsLifecycle = exports.DraupnirNewsHelper = exports.DraupnirNewsBlob = exports.DraupnirNewsItem = void 0;
const fs_1 = require("fs");
const typescript_result_1 = require("@gnuxie/typescript-result");
const typebox_1 = require("@sinclair/typebox");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const path_1 = __importDefault(require("path"));
const log = new matrix_protection_suite_1.Logger("DraupnirNews");
exports.DraupnirNewsItem = typebox_1.Type.Object({
    news_id: typebox_1.Type.String({
        description: "An identifier that can be persisted for an item of news.",
    }),
    matrix_event_content: typebox_1.Type.Union([matrix_protection_suite_1.MessageContent], {
        description: "Matrix event content for the news item that can be sent",
    }),
});
exports.DraupnirNewsBlob = typebox_1.Type.Object({
    news: typebox_1.Type.Array(exports.DraupnirNewsItem),
});
exports.DraupnirNewsHelper = Object.freeze({
    mergeSources(...blobs) {
        return [
            ...new Map(blobs
                .reduce((acc, blob) => [...acc, ...blob.news], [])
                .map((item) => [item.news_id, item])).values(),
        ];
    },
    removeSeenNews(news, seenNewsIDs) {
        return news.filter((item) => !seenNewsIDs.has(item.news_id));
    },
    removeUnseenNews(news, seenNewsIDs) {
        return news.filter((item) => seenNewsIDs.has(item.news_id));
    },
});
/**
 * This class manages requesting news from upstream, notifying, and storing
 * the newly seen news. Once seen news is updated, the instance should be
 * disposed.
 */
class DraupnirNewsLifecycle {
    constructor(seenNewsIDs, localNews, storeNews, fetchRemoteNews, notifyNewsItem) {
        this.seenNewsIDs = seenNewsIDs;
        this.localNews = localNews;
        this.storeNews = storeNews;
        this.fetchRemoteNews = fetchRemoteNews;
        this.notifyNewsItem = notifyNewsItem;
        // nothing to do.
    }
    async checkForNews() {
        const remoteNews = await this.fetchRemoteNews();
        if ((0, matrix_protection_suite_1.isError)(remoteNews)) {
            log.error("Unable to fetch news blob", remoteNews.error);
            // fall through, we still want to be able to show filesystem news.
        }
        const allNews = exports.DraupnirNewsHelper.mergeSources(this.localNews, (0, typescript_result_1.isOk)(remoteNews) ? remoteNews.ok : { news: [] });
        const unseenNews = exports.DraupnirNewsHelper.removeSeenNews(allNews, this.seenNewsIDs);
        if (unseenNews.length === 0) {
            return;
        }
        const notifiedNews = exports.DraupnirNewsHelper.removeUnseenNews(allNews, this.seenNewsIDs);
        for (const item of unseenNews) {
            const sendResult = await this.notifyNewsItem(item);
            if ((0, matrix_protection_suite_1.isError)(sendResult)) {
                log.error("Unable to notify of news item");
            }
            else {
                notifiedNews.push(item);
            }
        }
        const updateResult = await this.storeNews(notifiedNews);
        if ((0, matrix_protection_suite_1.isError)(updateResult)) {
            log.error("Unable to update stored news", updateResult.error);
            return;
        }
    }
}
exports.DraupnirNewsLifecycle = DraupnirNewsLifecycle;
const FSNews = (() => {
    const content = JSON.parse((0, fs_1.readFileSync)(path_1.default.join(__dirname, "./news.json"), "utf8"));
    return matrix_protection_suite_1.Value.Decode(exports.DraupnirNewsBlob, content).expect("File system news should match the schema");
})();
async function fetchNews(newsURL) {
    log.debug("Fetching remote news", newsURL);
    return await fetch(newsURL, {
        method: "GET",
        headers: {
            Accept: "application/json",
        },
    })
        .then((response) => response.json())
        .then((json) => matrix_protection_suite_1.Value.Decode(exports.DraupnirNewsBlob, json), (error) => matrix_protection_suite_1.ActionException.Result("unable to fetch news", {
        exception: error,
        exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown,
    }));
}
/**
 * This class schedules when to request news from the upstream repository.
 *
 * Lifecycle:
 * - unregisterListeners MUST be called when the parent protection is disposed.
 */
class DraupnirNewsReader {
    constructor(lifecycle, requestIntervalMS) {
        this.lifecycle = lifecycle;
        this.requestIntervalMS = requestIntervalMS;
        this.newsGate = new matrix_protection_suite_1.StandardTimedGate(this.requestNews.bind(this), this.requestIntervalMS);
        this.newsGate.enqueueOpen();
        this.requestLoop = this.createRequestLoop();
    }
    createRequestLoop() {
        return new matrix_protection_suite_1.ConstantPeriodBatch(() => {
            this.newsGate.enqueueOpen();
            this.requestLoop = this.createRequestLoop();
        }, this.requestIntervalMS);
    }
    async requestNews() {
        await this.lifecycle.checkForNews();
    }
    unregisterListeners() {
        this.newsGate.destroy();
        this.requestLoop.cancel();
    }
}
exports.DraupnirNewsReader = DraupnirNewsReader;
// Seen news gets cleaned up by storing the merged file system and remote
// news items which have been notified.
exports.DraupnirNewsProtectionSettings = typebox_1.Type.Object({
    seenNews: typebox_1.Type.Array(typebox_1.Type.String(), {
        default: [],
        uniqueItems: true,
        description: "Any news items that have been seen by the protection.",
    }),
}, {
    title: "DraupnirNewsProtectionSettings",
});
class DraupnirNews extends matrix_protection_suite_1.AbstractProtection {
    constructor(description, lifetime, capabilities, protectedRoomsSet, settings, draupnir) {
        super(description, lifetime, capabilities, protectedRoomsSet, {});
        this.settings = settings;
        this.draupnir = draupnir;
        this.newsReader = new DraupnirNewsReader(new DraupnirNewsLifecycle(new Set(this.settings.seenNews), FSNews, this.updateNews.bind(this), () => fetchNews(this.draupnir.config.draupnirNewsURL), (item) => this.draupnir.clientPlatform
            .toRoomMessageSender()
            .sendMessage(this.draupnir.managementRoomID, item.matrix_event_content)), 4.32e7 // 12 hours
        );
    }
    async updateNews(allNews) {
        const newSettings = this.description.protectionSettings.toMirror().setValue(this.settings, "seenNews", allNews.map((item) => item.news_id));
        if ((0, matrix_protection_suite_1.isError)(newSettings)) {
            return newSettings.elaborate("Unable to set protection settings");
        }
        const result = await this.protectedRoomsSet.protections.changeProtectionSettings(this.description, this.protectedRoomsSet, this.draupnir, newSettings.ok);
        if ((0, matrix_protection_suite_1.isError)(result)) {
            return result.elaborate("Unable to change protection settings");
        }
        return (0, typescript_result_1.Ok)(undefined);
    }
    handleProtectionDisable() {
        this.newsReader.unregisterListeners();
    }
}
exports.DraupnirNews = DraupnirNews;
(0, matrix_protection_suite_1.describeProtection)({
    name: DraupnirNews.name,
    description: "Provides news about the Draupnir project.",
    capabilityInterfaces: {},
    defaultCapabilities: {},
    configSchema: exports.DraupnirNewsProtectionSettings,
    async factory(description, lifetime, protectedRoomsSet, draupnir, capabilities, settings) {
        return (0, matrix_protection_suite_1.allocateProtection)(lifetime, new DraupnirNews(description, lifetime, capabilities, protectedRoomsSet, settings, draupnir));
    },
});
//# sourceMappingURL=DraupnirNews.js.map