"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.HomeserverUserPurgingDeactivate = void 0;
exports.isUserAccountRestricted = isUserAccountRestricted;
// Uses the user details api to check if the user is already locked or whatever
// calls the redact thing
// polls until it's done
// deactivates the user
// this service needs to be attached to draupnir but can be referenced
// in consequences.
const typescript_result_1 = require("@gnuxie/typescript-result");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const deactivateUser_1 = require("./deactivateUser");
const log = new matrix_protection_suite_1.Logger("HomeserverUserPurgingDeactivate");
function isUserAccountRestricted(details) {
    return (details.deactivated ||
        details.shadow_banned ||
        details.locked ||
        details.deactivated);
}
class HomeserverUserPurgingDeactivate {
    constructor(synapseAdminClient, userAuditLog) {
        this.synapseAdminClient = synapseAdminClient;
        this.userAuditLog = userAuditLog;
        this.checkProgress = async function (rawEntries) {
            for (const [userID, details] of rawEntries) {
                const redactionStatus = await this.synapseAdminClient.getUserRedactionStatus(details.redactionID);
                if ((0, typescript_result_1.isError)(redactionStatus)) {
                    log.error("Unable to query the status of a redaction for a user", userID, details.redactionID);
                }
                else if (redactionStatus.ok?.status === "active" ||
                    redactionStatus.ok?.status === "scheduled") {
                    this.batcher.add(userID, details);
                    continue;
                }
                const deactivateResult = await (0, deactivateUser_1.deactivateUser)(userID, this.synapseAdminClient, this.userAuditLog, {
                    rule: details.policy,
                    sender: details.sender,
                });
                if ((0, typescript_result_1.isError)(deactivateResult)) {
                    log.error("Unable to deactivate user", userID);
                    continue;
                }
            }
        }.bind(this);
        this.batcher = new matrix_protection_suite_1.StandardBatcher(() => new matrix_protection_suite_1.ConstantPeriodItemBatch(this.checkProgress, { waitPeriodMS: 20_000 }));
    }
    async beginPurgeUser(userID, { rule, sender, }) {
        const userDetails = await this.synapseAdminClient.getUserDetails(userID);
        if ((0, typescript_result_1.isError)(userDetails)) {
            return userDetails.elaborate("Should be able to fetch user details before purging");
        }
        // We make sure we have shadow banned their account while we redact.
        // This is to make sure they don't send anymore while we decomission the account.
        if (!userDetails.ok || !isUserAccountRestricted(userDetails.ok)) {
            const shadowBanResult = await this.synapseAdminClient.shadowBanUser(userID);
            if ((0, typescript_result_1.isError)(shadowBanResult)) {
                log.error("Failed to shadowban user before starting pre-deactivation redaction", userID);
            }
        }
        const redactionStartResult = await this.synapseAdminClient.redactUser(userID);
        if ((0, typescript_result_1.isError)(redactionStartResult)) {
            return redactionStartResult.elaborate("Failed to begin pre-deactivation redaction process");
        }
        this.batcher.add(userID, {
            policy: rule,
            sender,
            redactionID: redactionStartResult.ok.redact_id,
        });
        return (0, typescript_result_1.Ok)(undefined);
    }
}
exports.HomeserverUserPurgingDeactivate = HomeserverUserPurgingDeactivate;
//# sourceMappingURL=HomeserverUserPurgingDeactivate.js.map