"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardDiscoveredRoomStore = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const events_1 = __importDefault(require("events"));
const log = new matrix_protection_suite_1.Logger("DiscoveredRoomStore");
/**
 * Currently the store is polluted because of https://github.com/element-hq/synapse/issues/18563
 * Because of the fallout of this bug, there will be room details polluted in the following way:
 *
 * 1. empty string for `server`.
 * 2. empty string for `creator`.
 *
 * To cleanup, we probably need to destroy the details table entirely
 * and create a table for the server responsible for creating the room derived
 * from the room_id.
 * Then later if we find a way of obtaining the creator, we can store that
 * instead.
 *
 * The reason why this didn't happen in the first place is because we thought
 * that the creator would always be available and we would always derive the
 * server from the creator.
 */
class StandardDiscoveredRoomStore extends events_1.default {
    constructor(hashStore, roomDetailsProvider) {
        super();
        this.hashStore = hashStore;
        this.roomDetailsProvider = roomDetailsProvider;
        this.discoveredRooms = new Set();
    }
    async addRoomDetails(discoveredRoomsRecords) {
        const discoveredRooms = [];
        for (const { roomID, details: providedDetails } of discoveredRoomsRecords) {
            const detailsResult = providedDetails === undefined
                ? await this.roomDetailsProvider.getRoomDetails(roomID)
                : (0, typescript_result_1.Ok)(providedDetails);
            if ((0, typescript_result_1.isError)(detailsResult)) {
                log.error("Error fetching details for a discovered room", roomID, detailsResult.error);
            }
            const details = (0, typescript_result_1.isError)(detailsResult)
                ? { room_id: roomID }
                : detailsResult.ok;
            discoveredRooms.push(details);
            if (details.creator === undefined) {
                continue; // no point persisting details that don't have a creator.
            }
            const storeResult = await this.hashStore.storeRoomIdentification({
                creator: details.creator,
                roomID: details.room_id,
                server: (0, matrix_basic_types_1.userServerName)(details.creator),
            });
            if ((0, typescript_result_1.isError)(storeResult)) {
                log.error("Error storing room details for a room", roomID, details, storeResult);
            }
        }
        return (0, typescript_result_1.Ok)(discoveredRooms);
    }
    async checkRoomsDiscovered(roomsToCheck) {
        const entries = roomsToCheck.map((entry) => entry.roomID);
        const storeResult = await this.hashStore.storeUndiscoveredRooms(entries);
        if ((0, typescript_result_1.isError)(storeResult)) {
            return storeResult.elaborate("Unxpected error while trying to store undiscovered rooms");
        }
        const detailsResult = await this.addRoomDetails(roomsToCheck);
        if ((0, typescript_result_1.isError)(detailsResult)) {
            return detailsResult.elaborate("Unxpected error while trying to add room details");
        }
        for (const details of detailsResult.ok) {
            this.discoveredRooms.add(details.room_id);
            this.emit("RoomDiscovered", details);
        }
        return detailsResult;
    }
    isRoomDiscovered(roomID) {
        return this.discoveredRooms.has(roomID);
    }
}
exports.StandardDiscoveredRoomStore = StandardDiscoveredRoomStore;
//# sourceMappingURL=DiscoveredRoomStore.js.map