"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.RoomTakedownProtection = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const RoomTakedown_1 = require("./RoomTakedown");
const SynapseAdminRoomTakedown_1 = require("../../capabilities/SynapseAdminRoomTakedown/SynapseAdminRoomTakedown");
const typescript_result_1 = require("@gnuxie/typescript-result");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
const typebox_1 = require("@sinclair/typebox");
const RoomDiscoveryRenderer_1 = require("./RoomDiscoveryRenderer");
const NotificationRoom_1 = require("../NotificationRoom/NotificationRoom");
const matrix_bot_sdk_1 = require("matrix-bot-sdk");
const SynapseHTTPAntispamRoomExplorer_1 = require("./SynapseHTTPAntispamRoomExplorer");
const SynapseRoomListRoomExplorerer_1 = require("./SynapseRoomListRoomExplorerer");
const DiscoveredRoomStore_1 = require("./DiscoveredRoomStore");
// FIXME: I don't like that the exploreres are tied to this protection anymore!
// I think they should be distinct protections. AAAAAAAAAAAAAAAAAAAAAAAAaaa
// But imagine that we did want the discovery event emitter... how would that
// work with protections?
// I'll tell you. The consumer would have to name each protection it wants
// and find them and attach listeners. And then it'd also need an event
// to show when the source gets destroyed.
// hmm we can still do it if we just wack the RoomDiscovery implementation
// onto the draupnir class.
const log = new matrix_protection_suite_1.Logger("RoomTakedownProtection");
const RoomTakedownProtectionSettings = typebox_1.Type.Object({
    discoveryNotificationMembershipThreshold: typebox_1.Type.Integer({
        default: 20,
        description: "The number of members required in the room for it to appear in the notification. This is to prevent showing direct messages or small rooms that could be too much of an invasion of privacy. We don't have access to enough information to determine this a better way.",
    }),
    // There needs to be a transform for room references
    discoveryNotificationRoom: typebox_1.Type.Optional(typebox_1.Type.Union([matrix_protection_suite_1.StringRoomIDSchema, typebox_1.Type.Undefined()], {
        default: undefined,
        description: "The room where notifications should be sent.",
    })),
    discoveryNotificationEnabled: typebox_1.Type.Boolean({
        default: false,
        description: "Wether to send notifications for newly discovered rooms from the homerserver.",
    }),
    roomListScanIntervalMS: typebox_1.Type.Integer({
        default: 30 * 60_000,
        description: "How frequently to scan the entire list of rooms synapse is joined to. This is a huge operation",
    }),
    roomListScanCooldownMS: typebox_1.Type.Integer({
        default: 5 * 60_000,
        description: "The minimum amount of time the protection should wait between each scan of the room list. If you are using synapse-http-antispam this should be quite a long time.",
    }),
}, { title: "RoomTakedownProtectionSettings" });
class RoomTakedownProtection extends matrix_protection_suite_1.AbstractProtection {
    constructor(description, lifetime, capabilities, protectedRoomsSet, auditLog, hashStore, automaticallyRedactForReasons, roomMessageSender, discoveryNotificationEnabled, discoveryNotificationMembershipThreshold, discoveryNotificationRoom, roomExplorers, roomDiscovery) {
        super(description, lifetime, capabilities, protectedRoomsSet, {});
        this.roomMessageSender = roomMessageSender;
        this.discoveryNotificationEnabled = discoveryNotificationEnabled;
        this.discoveryNotificationMembershipThreshold = discoveryNotificationMembershipThreshold;
        this.discoveryNotificationRoom = discoveryNotificationRoom;
        this.roomExplorers = roomExplorers;
        this.roomDiscovery = roomDiscovery;
        this.roomDiscoveryListener = function (details) {
            if ((details.joined_members ?? 0) <
                this.discoveryNotificationMembershipThreshold) {
                return;
            }
            void (0, matrix_protection_suite_1.Task)((async () => {
                const sendResult = await (0, mps_interface_adaptor_1.sendMatrixEventsFromDeadDocument)(this.roomMessageSender, this.discoveryNotificationRoom, (0, mps_interface_adaptor_1.wrapInRoot)((0, RoomDiscoveryRenderer_1.renderDiscoveredRoom)(details)), {});
                if ((0, typescript_result_1.isError)(sendResult)) {
                    log.error("Error sending a notification about a discovered room", details.room_id, sendResult.error);
                }
            })());
        }.bind(this);
        this.roomTakedown = new RoomTakedown_1.StandardRoomTakedown(auditLog, capabilities.roomTakedownCapability);
        void (0, matrix_protection_suite_1.Task)(this.roomTakedown.checkAllRooms(this.protectedRoomsSet.watchedPolicyRooms.currentRevision));
        if (this.discoveryNotificationEnabled) {
            this.roomDiscovery?.on("RoomDiscovery", this.roomDiscoveryListener);
        }
    }
    handlePolicyChange(revision, changes) {
        return this.roomTakedown.handlePolicyChange(revision, changes);
    }
    handleProtectionDisable() {
        this.roomExplorers.forEach((explorer) => {
            explorer.unregisterListeners();
        });
        this.roomDiscovery?.off("RoomDiscovery", this.roomDiscoveryListener);
    }
}
exports.RoomTakedownProtection = RoomTakedownProtection;
(0, matrix_protection_suite_1.describeProtection)({
    name: RoomTakedownProtection.name,
    description: `A protection to shutdown rooms matching policies from watched lists`,
    capabilityInterfaces: {
        roomTakedownCapability: "RoomTakedownCapability",
    },
    defaultCapabilities: {
        roomTakedownCapability: SynapseAdminRoomTakedown_1.SynapseAdminRoomTakedownCapability.name,
    },
    configSchema: RoomTakedownProtectionSettings,
    async factory(description, lifetime, protectedRoomsSet, draupnir, capabilitySet, settings) {
        if (settings.discoveryNotificationEnabled &&
            settings.discoveryNotificationRoom === undefined) {
            // FIXME: The type parameters are really fucked for the protection system
            // and that needs fixing. The problem is that the protection system was written
            // before we knew how to do this properly.
            return (await NotificationRoom_1.NotificationRoomCreator.createNotificationRoomFromDraupnir(draupnir, description, settings, "discoveryNotificationRoom", "Room Discovery Notification", log));
        }
        if (draupnir.stores.hashStore === undefined ||
            draupnir.stores.roomAuditLog === undefined) {
            return typescript_result_1.ResultError.Result("This protection requires a hash store and audit log to be available to draupnir, and they are not in your configuration.");
        }
        const roomDetailsProvider = draupnir.synapseAdminClient
            ? new SynapseAdminRoomTakedown_1.SynapseAdminRoomDetailsProvider(draupnir.synapseAdminClient)
            : undefined;
        const roomDiscovery = roomDetailsProvider
            ? new DiscoveredRoomStore_1.StandardDiscoveredRoomStore(draupnir.stores.hashStore, roomDetailsProvider)
            : undefined;
        const synapseHTTPAntispamRoomExplorer = draupnir.synapseHTTPAntispam && roomDiscovery
            ? new SynapseHTTPAntispamRoomExplorer_1.SynapseHTTPAntispamRoomExplorer(draupnir.synapseHTTPAntispam, roomDiscovery)
            : undefined;
        const synapseRoomListRoomExplorer = draupnir.synapseAdminClient && roomDiscovery
            ? new SynapseRoomListRoomExplorerer_1.SynapseRoomListRoomExplorer(settings.roomListScanCooldownMS, settings.roomListScanIntervalMS, new SynapseRoomListRoomExplorerer_1.SynapseRoomListScanner(roomDiscovery, draupnir.synapseAdminClient))
            : undefined;
        const roomExploreres = [];
        if (synapseHTTPAntispamRoomExplorer) {
            roomExploreres.push(synapseHTTPAntispamRoomExplorer);
        }
        if (synapseRoomListRoomExplorer) {
            roomExploreres.push(synapseRoomListRoomExplorer);
        }
        return (0, matrix_protection_suite_1.allocateProtection)(lifetime, new RoomTakedownProtection(description, lifetime, capabilitySet, protectedRoomsSet, draupnir.stores.roomAuditLog, draupnir.stores.hashStore, draupnir.config.automaticallyRedactForReasons.map((reason) => new matrix_bot_sdk_1.MatrixGlob(reason)), draupnir.clientPlatform.toRoomMessageSender(), settings.discoveryNotificationEnabled, settings.discoveryNotificationMembershipThreshold, settings.discoveryNotificationRoom ?? draupnir.managementRoomID, roomExploreres, roomDiscovery));
    },
});
//# sourceMappingURL=RoomTakedownProtection.js.map