const tslib = require('tslib');
const core = require('@sentry/core');
const utils = require('@sentry/utils');
const electron = require('electron');
const util = require('util');
const anr = require('./anr.js');
const electronNormalize = require('./electron-normalize.js');
const ipc = require('../common/ipc.js');
const normalize = require('../common/normalize.js');
const mode = require('../common/mode.js');
const merge = require('../common/merge.js');

let KNOWN_RENDERERS;
let WINDOW_ID_TO_WEB_CONTENTS;
function newProtocolRenderer() {
    return tslib.__awaiter(this, void 0, void 0, function* () {
        KNOWN_RENDERERS = KNOWN_RENDERERS || new Set();
        WINDOW_ID_TO_WEB_CONTENTS = WINDOW_ID_TO_WEB_CONTENTS || new Map();
        for (const wc of electron.webContents.getAllWebContents()) {
            const wcId = wc.id;
            if (KNOWN_RENDERERS.has(wcId)) {
                continue;
            }
            if (!wc.isDestroyed()) {
                try {
                    const windowId = yield wc.executeJavaScript('window.__SENTRY_RENDERER_ID__');
                    if (windowId) {
                        KNOWN_RENDERERS.add(wcId);
                        WINDOW_ID_TO_WEB_CONTENTS.set(windowId, wcId);
                        wc.once('destroyed', () => {
                            KNOWN_RENDERERS === null || KNOWN_RENDERERS === void 0 ? void 0 : KNOWN_RENDERERS.delete(wcId);
                            WINDOW_ID_TO_WEB_CONTENTS === null || WINDOW_ID_TO_WEB_CONTENTS === void 0 ? void 0 : WINDOW_ID_TO_WEB_CONTENTS.delete(windowId);
                        });
                    }
                }
                catch (_) {
                    // ignore
                }
            }
        }
    });
}
function captureEventFromRenderer(options, event, attachments, contents) {
    var _a, _b, _c, _d;
    const process = contents ? ((_a = options === null || options === void 0 ? void 0 : options.getRendererName) === null || _a === void 0 ? void 0 : _a.call(options, contents)) || 'renderer' : 'renderer';
    // Ensure breadcrumbs are empty as they sent via scope updates
    event.breadcrumbs = event.breadcrumbs || [];
    // Remove the environment as it defaults to 'production' and overwrites the main process environment
    delete event.environment;
    // Remove the SDK info as we want the Electron SDK to be the one reporting the event
    (_b = event.sdk) === null || _b === void 0 ? true : delete _b.name;
    (_c = event.sdk) === null || _c === void 0 ? true : delete _c.version;
    (_d = event.sdk) === null || _d === void 0 ? true : delete _d.packages;
    core.captureEvent(merge.mergeEvents(event, { tags: { 'event.process': process } }), { attachments });
}
function handleEvent(options, jsonEvent, contents) {
    let event;
    try {
        event = JSON.parse(jsonEvent);
    }
    catch (_a) {
        utils.logger.warn('sentry-electron received an invalid event message');
        return;
    }
    captureEventFromRenderer(options, event, [], contents);
}
function eventFromEnvelope(envelope) {
    let event;
    const attachments = [];
    utils.forEachEnvelopeItem(envelope, (item, type) => {
        if (type === 'event' || type === 'transaction') {
            event = Array.isArray(item) ? item[1] : undefined;
        }
        else if (type === 'attachment') {
            const [headers, data] = item;
            attachments.push({
                filename: headers.filename,
                attachmentType: headers.attachment_type,
                contentType: headers.content_type,
                data,
            });
        }
    });
    return event ? [event, attachments] : undefined;
}
function handleEnvelope(options, env, contents) {
    var _a, _b;
    const envelope = utils.parseEnvelope(env, new util.TextEncoder(), new util.TextDecoder());
    const eventAndAttachments = eventFromEnvelope(envelope);
    if (eventAndAttachments) {
        const [event, attachments] = eventAndAttachments;
        captureEventFromRenderer(options, event, attachments, contents);
    }
    else {
        const normalizedEnvelope = normalize.normalizeUrlsInReplayEnvelope(envelope, electron.app.getAppPath());
        // Pass other types of envelope straight to the transport
        void ((_b = (_a = core.getCurrentHub().getClient()) === null || _a === void 0 ? void 0 : _a.getTransport()) === null || _b === void 0 ? void 0 : _b.send(normalizedEnvelope));
    }
}
/** Is object defined and has keys */
function hasKeys(obj) {
    return obj != undefined && Object.keys(obj).length > 0;
}
/**
 * Handle scope updates from renderer processes
 */
function handleScope(options, jsonScope) {
    let rendererScope;
    try {
        rendererScope = JSON.parse(jsonScope);
    }
    catch (_a) {
        utils.logger.warn('sentry-electron received an invalid scope message');
        return;
    }
    const sentScope = core.Scope.clone(rendererScope);
    /* eslint-disable @typescript-eslint/no-unsafe-member-access */
    core.configureScope((scope) => {
        if (hasKeys(sentScope._user)) {
            scope.setUser(sentScope._user);
        }
        if (hasKeys(sentScope._tags)) {
            scope.setTags(sentScope._tags);
        }
        if (hasKeys(sentScope._extra)) {
            scope.setExtras(sentScope._extra);
        }
        for (const attachment of sentScope._attachments || []) {
            scope.addAttachment(attachment);
        }
        const breadcrumb = sentScope._breadcrumbs.pop();
        if (breadcrumb) {
            scope.addBreadcrumb(breadcrumb, (options === null || options === void 0 ? void 0 : options.maxBreadcrumbs) || 100);
        }
    });
    /* eslint-enable @typescript-eslint/no-unsafe-member-access */
}
/** Enables Electron protocol handling */
function configureProtocol(options) {
    if (electron.app.isReady()) {
        throw new utils.SentryError("Sentry SDK should be initialized before the Electron app 'ready' event is fired");
    }
    electron.protocol.registerSchemesAsPrivileged([
        {
            scheme: ipc.PROTOCOL_SCHEME,
            privileges: { bypassCSP: true, corsEnabled: true, supportFetchAPI: true, secure: true },
        },
    ]);
    const rendererStatusChanged = anr.createRendererAnrStatusHook();
    electronNormalize.whenAppReady
        .then(() => {
        for (const sesh of options.getSessions()) {
            electronNormalize.registerProtocol(sesh.protocol, ipc.PROTOCOL_SCHEME, (request) => {
                const getWebContents = () => {
                    const webContentsId = request.windowId ? WINDOW_ID_TO_WEB_CONTENTS === null || WINDOW_ID_TO_WEB_CONTENTS === void 0 ? void 0 : WINDOW_ID_TO_WEB_CONTENTS.get(request.windowId) : undefined;
                    return webContentsId ? electron.webContents.fromId(webContentsId) : undefined;
                };
                const data = request.body;
                if (request.url.startsWith(`${ipc.PROTOCOL_SCHEME}://${ipc.IPCChannel.RENDERER_START}`)) {
                    void newProtocolRenderer();
                }
                else if (request.url.startsWith(`${ipc.PROTOCOL_SCHEME}://${ipc.IPCChannel.EVENT}`) && data) {
                    handleEvent(options, data.toString(), getWebContents());
                }
                else if (request.url.startsWith(`${ipc.PROTOCOL_SCHEME}://${ipc.IPCChannel.SCOPE}`) && data) {
                    handleScope(options, data.toString());
                }
                else if (request.url.startsWith(`${ipc.PROTOCOL_SCHEME}://${ipc.IPCChannel.ENVELOPE}`) && data) {
                    handleEnvelope(options, data, getWebContents());
                }
                else if (request.url.startsWith(`${ipc.PROTOCOL_SCHEME}://${ipc.IPCChannel.STATUS}`) && data) {
                    const contents = getWebContents();
                    if (contents) {
                        const status = JSON.parse(data.toString()).status;
                        rendererStatusChanged(status, contents);
                    }
                }
            });
        }
    })
        .catch((error) => utils.logger.error(error));
}
/**
 * Hooks IPC for communication with the renderer processes
 */
function configureClassic(options) {
    electron.ipcMain.on(ipc.IPCChannel.RENDERER_START, ({ sender }) => {
        const id = sender.id;
        // In older Electron, sender can be destroyed before this callback is called
        if (!sender.isDestroyed()) {
            // Keep track of renderers that are using IPC
            KNOWN_RENDERERS = KNOWN_RENDERERS || new Set();
            KNOWN_RENDERERS.add(id);
            sender.once('destroyed', () => {
                KNOWN_RENDERERS === null || KNOWN_RENDERERS === void 0 ? void 0 : KNOWN_RENDERERS.delete(id);
            });
        }
    });
    electron.ipcMain.on(ipc.IPCChannel.EVENT, ({ sender }, jsonEvent) => handleEvent(options, jsonEvent, sender));
    electron.ipcMain.on(ipc.IPCChannel.SCOPE, (_, jsonScope) => handleScope(options, jsonScope));
    electron.ipcMain.on(ipc.IPCChannel.ENVELOPE, ({ sender }, env) => handleEnvelope(options, env, sender));
    const rendererStatusChanged = anr.createRendererAnrStatusHook();
    electron.ipcMain.on(ipc.IPCChannel.STATUS, ({ sender }, status) => rendererStatusChanged(status, sender));
}
/** Sets up communication channels with the renderer */
function configureIPC(options) {
    if (!electronNormalize.supportsFullProtocol() && options.ipcMode === mode.IPCMode.Protocol) {
        throw new utils.SentryError('IPCMode.Protocol is only supported in Electron >= v5');
    }
    // eslint-disable-next-line no-bitwise
    if (electronNormalize.supportsFullProtocol() && (options.ipcMode & mode.IPCMode.Protocol) > 0) {
        configureProtocol(options);
    }
    // eslint-disable-next-line no-bitwise
    if ((options.ipcMode & mode.IPCMode.Classic) > 0) {
        configureClassic(options);
    }
}

exports.configureIPC = configureIPC;
//# sourceMappingURL=ipc.js.map
