/*=========================================================================
 *
 *  Copyright NumFOCUS
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *         https://www.apache.org/licenses/LICENSE-2.0.txt
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *=========================================================================*/
#ifndef itkQuadEdgeMeshEulerOperatorSplitFacetFunction_hxx
#define itkQuadEdgeMeshEulerOperatorSplitFacetFunction_hxx


namespace itk
{
template <typename TMesh, typename TQEType>
auto
QuadEdgeMeshEulerOperatorSplitFacetFunction<TMesh, TQEType>::Evaluate(QEType * h, QEType * g) -> OutputType
{
  //
  //  g->Dest() ---<----- X                    destPid  --------- X        //
  //         /     g       \                         / \           \       //
  //        /               \                   dLnext  \           \      //
  //       /                 \                     /     \           \     //
  //      /                   \                   v       ^           \    //
  //     /      g->Left()      \                 /         \           \   //
  //    X           =           X               X        newEdge        X  //
  //     \      h->Left()      /                 \           \         /   //
  //      \                   /                   \           \       ^    //
  //       \                 /                     \           \     /     //
  //        \               /                       \           \ oLnext   //
  //         \       h     /                         \           \ /       //
  //          X ----->--- h->Dest()                   X --------- orgPid   //
  //

  if (!h || !g)
  {
    itkDebugMacro("At least one of the Input is not an edge.");
    return (QEType *)nullptr;
  }

  if (!this->m_Mesh)
  {
    itkDebugMacro("No mesh present.");
    return (QEType *)nullptr;
  }

  if (h == g)
  {
    itkDebugMacro("Provided edges should be different.");
    return (QEType *)nullptr;
  }

  if (h->GetLeft() != g->GetLeft())
  {
    itkDebugMacro("The edges are not around the same face.");
    return (QEType *)nullptr;
  }

  if ((h->GetLnext() == g) || (g->GetLnext() == h))
  {
    itkDebugMacro("Provided edges should NOT be consecutive.");
    return (QEType *)nullptr;
  }

  using VertexRefType = typename MeshType::VertexRefType;

  this->m_Mesh->DeleteFace(h->GetLeft());
  const VertexRefType orgPid = h->GetDestination();
  const VertexRefType destPid = g->GetDestination();

  // Create an new isolated edge and set its geometry:
  auto *   newEdge = new EdgeCellType;
  QEType * newEdgeGeom = newEdge->GetQEGeom();

  // see the code of e.g. AddFace
  newEdgeGeom->SetOrigin(orgPid);
  newEdgeGeom->SetDestination(destPid);

  // Insert newEdge at Org
  QEType * oLnext = h->GetLnext();
  oLnext->InsertAfterNextBorderEdgeWithUnsetLeft(newEdgeGeom);
  // Insert newEdge at Dest
  QEType * dLnext = g->GetLnext();
  dLnext->InsertAfterNextBorderEdgeWithUnsetLeft(newEdgeGeom->GetSym());

  // Add the new edge to the container
  this->m_Mesh->PushOnContainer(newEdge);

  // Build two new faces
  this->m_Mesh->AddFace(h);
  this->m_Mesh->AddFace(g);
  this->m_Mesh->Modified();
  return newEdgeGeom;
}

} // end namespace itk

#endif
