"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const base_command_1 = require("./base-command");
const app_1 = require("./app");
const { renderCommandHelp } = require('./help-utils.js');
const locale_1 = require("@joplin/lib/locale");
const { cliUtils } = require('./cli-utils.js');
class Command extends base_command_1.default {
    usage() {
        return 'help [command]';
    }
    description() {
        return (0, locale_1._)('Displays usage information.');
    }
    allCommands() {
        const commands = (0, app_1.default)().commands((0, app_1.default)().uiType());
        const output = [];
        for (const n in commands) {
            if (!commands.hasOwnProperty(n))
                continue;
            const command = commands[n];
            if (command.hidden())
                continue;
            if (!command.enabled())
                continue;
            output.push(command);
        }
        output.sort((a, b) => (a.name() < b.name() ? -1 : +1));
        return output;
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async action(args) {
        const stdoutWidth = (0, app_1.default)().commandStdoutMaxWidth();
        if (args.command === 'shortcuts' || args.command === 'keymap') {
            this.stdout((0, locale_1._)('For information on how to customise the shortcuts please visit %s', 'https://joplinapp.org/help/apps/terminal#shortcuts'));
            this.stdout('');
            if ((0, app_1.default)()
                .gui()
                .isDummy()) {
                throw new Error((0, locale_1._)('Shortcuts are not available in CLI mode.'));
            }
            const keymap = (0, app_1.default)()
                .gui()
                .keymap();
            const rows = [];
            for (let i = 0; i < keymap.length; i++) {
                const item = keymap[i];
                const keys = item.keys.map((k) => (k === ' ' ? '(SPACE)' : k));
                rows.push([keys.join(', '), item.command]);
            }
            cliUtils.printArray(this.stdout.bind(this), rows);
        }
        else if (args.command === 'all') {
            const commands = this.allCommands();
            const output = commands.map(c => renderCommandHelp(c));
            this.stdout(output.join('\n\n'));
        }
        else if (args.command) {
            const command = (0, app_1.default)().findCommandByName(args['command']);
            if (!command)
                throw new Error((0, locale_1._)('Cannot find "%s".', args.command));
            this.stdout(renderCommandHelp(command, stdoutWidth));
        }
        else {
            const commandNames = this.allCommands().map(a => a.name());
            this.stdout((0, locale_1._)('Type `help [command]` for more information about a command; or type `help all` for the complete usage information.'));
            this.stdout('');
            this.stdout((0, locale_1._)('The possible commands are:'));
            this.stdout('');
            this.stdout(commandNames.join(', '));
            this.stdout('');
            this.stdout((0, locale_1._)('In any command, a note or notebook can be referred to by title or ID, or using the shortcuts `$n` or `$b` for, respectively, the currently selected note or notebook. `$c` can be used to refer to the currently selected item.'));
            this.stdout('');
            this.stdout((0, locale_1._)('To move from one pane to another, press Tab or Shift+Tab.'));
            this.stdout((0, locale_1._)('Use the arrows and page up/down to scroll the lists and text areas (including this console).'));
            this.stdout((0, locale_1._)('To maximise/minimise the console, press "tc".'));
            this.stdout((0, locale_1._)('To enter command line mode, press ":"'));
            this.stdout((0, locale_1._)('To exit command line mode, press ESCAPE'));
            this.stdout((0, locale_1._)('For the list of keyboard shortcuts and config options, type `help keymap`'));
        }
        (0, app_1.default)()
            .gui()
            .showConsole();
        (0, app_1.default)()
            .gui()
            .maximizeConsole();
    }
}
module.exports = Command;
//# sourceMappingURL=command-help.js.map