"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Setting_1 = require("../models/Setting");
const test_utils_1 = require("../testing/test-utils");
const fs_extra_1 = require("fs-extra");
const Logger_1 = require("@joplin/utils/Logger");
const types_1 = require("../services/profileConfig/types");
const profileConfig_1 = require("../services/profileConfig");
const initProfile_1 = require("../services/profileConfig/initProfile");
const PluginService_1 = require("../services/plugins/PluginService");
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
async function loadSettingsFromFile() {
    return JSON.parse(await (0, fs_extra_1.readFile)(Setting_1.default.settingFilePath, 'utf8'));
}
const loadDefaultProfileSettings = async () => {
    return JSON.parse(await (0, fs_extra_1.readFile)(`${Setting_1.default.value('rootProfileDir')}/settings-1.json`, 'utf8'));
};
const switchToSubProfileSettings = async () => {
    await Setting_1.default.reset();
    const rootProfileDir = Setting_1.default.value('profileDir');
    const profileConfigPath = `${rootProfileDir}/profiles.json`;
    let profileConfig = (0, types_1.defaultProfileConfig)();
    const { newConfig, newProfile } = (0, profileConfig_1.createNewProfile)(profileConfig, 'Sub-profile');
    profileConfig = newConfig;
    profileConfig.currentProfileId = newProfile.id;
    await (0, profileConfig_1.saveProfileConfig)(profileConfigPath, profileConfig);
    const { profileDir } = await (0, initProfile_1.default)(rootProfileDir);
    await (0, fs_extra_1.mkdirp)(profileDir);
    await Setting_1.default.load();
};
describe('models/Setting', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should return only sub-values', (async () => {
        const settings = {
            'sync.5.path': 'http://example.com',
            'sync.5.username': 'testing',
        };
        let output = Setting_1.default.subValues('sync.5', settings);
        expect(output['path']).toBe('http://example.com');
        expect(output['username']).toBe('testing');
        output = Setting_1.default.subValues('sync.4', settings);
        expect('path' in output).toBe(false);
        expect('username' in output).toBe(false);
    }));
    it('should not fail when trying to load a key that no longer exist from the setting file', (async () => {
        // To handle the case where a setting value exists in the database but
        // the metadata has been removed in a new Joplin version.
        // https://github.com/laurent22/joplin/issues/5086
        Setting_1.default.setValue('sync.target', 9); // Saved to file
        await Setting_1.default.saveAll();
        const settingValues = await Setting_1.default.fileHandler.load();
        settingValues['itsgone'] = 'myvalue';
        await Setting_1.default.fileHandler.save(settingValues);
        await Setting_1.default.reset();
        await (0, test_utils_1.expectNotThrow)(async () => Setting_1.default.load());
        await (0, test_utils_1.expectThrow)(async () => Setting_1.default.value('itsgone'), 'unknown_key');
    }));
    it.each([
        Setting_1.SettingStorage.Database, Setting_1.SettingStorage.File,
    ])('should allow registering new settings dynamically (storage: %d)', (async (storage) => {
        await (0, test_utils_1.expectThrow)(async () => Setting_1.default.setValue('myCustom', '123'), 'unknown_key');
        await Setting_1.default.registerSetting('myCustom', {
            public: true,
            value: 'default',
            type: Setting_1.default.TYPE_STRING,
            storage,
        });
        expect(Setting_1.default.value('myCustom')).toBe('default');
        await (0, test_utils_1.expectNotThrow)(async () => Setting_1.default.setValue('myCustom', '123'));
        expect(Setting_1.default.value('myCustom')).toBe('123');
    }));
    it.each([Setting_1.SettingStorage.Database, Setting_1.SettingStorage.File])('should not clear old custom settings if not registered immediately', (async (storage) => {
        // In general the following should work:
        //
        // - Plugin register a new setting
        // - User set new value for setting
        // - Settings are saved
        // - => App restart
        // - Plugin does not register setting again
        // - Settings are loaded
        // - Settings are saved
        // - Plugin register setting again
        // - Previous value set by user should still be there.
        //
        // In other words, once a custom setting has been set we don't clear it
        // even if registration doesn't happen immediately. That allows for example
        // to delay setting registration without a risk for the custom setting to be deleted.
        await Setting_1.default.registerSetting('myCustom', {
            public: true,
            value: 'default',
            type: Setting_1.default.TYPE_STRING,
            storage,
        });
        Setting_1.default.setValue('myCustom', '123');
        await Setting_1.default.saveAll();
        await Setting_1.default.reset();
        await Setting_1.default.load();
        await Setting_1.default.registerSetting('myCustom', {
            public: true,
            value: 'default',
            type: Setting_1.default.TYPE_STRING,
            storage,
        });
        await Setting_1.default.saveAll();
        expect(Setting_1.default.value('myCustom')).toBe('123');
    }));
    it.each([Setting_1.SettingStorage.Database, Setting_1.SettingStorage.File])('should not clear old custom settings if not registered until restart', async (storage) => {
        const registerCustom = async () => {
            await Setting_1.default.registerSetting('myCustom', {
                public: true,
                value: 'test',
                type: Setting_1.default.TYPE_STRING,
                storage,
            });
        };
        await registerCustom();
        Setting_1.default.setValue('myCustom', 'test2');
        await Setting_1.default.saveAll();
        await Setting_1.default.reset();
        await Setting_1.default.load();
        // Change a file setting
        Setting_1.default.setValue('sync.target', 9);
        await Setting_1.default.saveAll();
        await Setting_1.default.reset();
        await Setting_1.default.load();
        await registerCustom();
        expect(Setting_1.default.value('myCustom')).toBe('test2');
    });
    it('should return values with correct type for custom settings', (async () => {
        await Setting_1.default.registerSetting('myCustom', {
            public: true,
            value: 123,
            type: Setting_1.default.TYPE_INT,
        });
        Setting_1.default.setValue('myCustom', 456);
        await Setting_1.default.saveAll();
        await Setting_1.default.reset();
        await Setting_1.default.load();
        await Setting_1.default.registerSetting('myCustom', {
            public: true,
            value: 123,
            type: Setting_1.default.TYPE_INT,
        });
        expect(typeof Setting_1.default.value('myCustom')).toBe('number');
        expect(Setting_1.default.value('myCustom')).toBe(456);
    }));
    it('should validate registered keys', (async () => {
        const md = {
            public: true,
            value: 'default',
            type: Setting_1.default.TYPE_STRING,
        };
        await (0, test_utils_1.expectThrow)(async () => await Setting_1.default.registerSetting('', md));
        await (0, test_utils_1.expectThrow)(async () => await Setting_1.default.registerSetting('no spaces', md));
        await (0, test_utils_1.expectThrow)(async () => await Setting_1.default.registerSetting('nospecialcharacters!!!', md));
        await (0, test_utils_1.expectThrow)(async () => await Setting_1.default.registerSetting('Robert\'); DROP TABLE Students;--', md));
        await (0, test_utils_1.expectNotThrow)(async () => await Setting_1.default.registerSetting('numbersareok123', md));
        await (0, test_utils_1.expectNotThrow)(async () => await Setting_1.default.registerSetting('so-ARE-dashes_123', md));
    }));
    it('should register new sections', (async () => {
        await Setting_1.default.registerSection('mySection', Setting_1.SettingSectionSource.Default, {
            label: 'My section',
        });
        expect(Setting_1.default.sectionNameToLabel('mySection')).toBe('My section');
    }));
    it('should save and load settings from file', (async () => {
        Setting_1.default.setValue('sync.target', 9); // Saved to file
        Setting_1.default.setValue('encryption.passwordCache', {}); // Saved to keychain or db
        Setting_1.default.setValue('plugins.states', { test: (0, PluginService_1.defaultPluginSetting)() }); // Always saved to db
        await Setting_1.default.saveAll();
        {
            const settings = await loadSettingsFromFile();
            expect(settings['sync.target']).toBe(9);
            expect(settings).not.toContain('encryption.passwordCache');
            expect(settings).not.toContain('plugins.states');
        }
        Setting_1.default.setValue('sync.target', 8);
        await Setting_1.default.saveAll();
        {
            const settings = await loadSettingsFromFile();
            expect(settings['sync.target']).toBe(8);
        }
    }));
    it('should not save to file if nothing has changed', (async () => {
        Setting_1.default.setValue('sync.mobileWifiOnly', true);
        await Setting_1.default.saveAll();
        {
            // Double-check that timestamp is indeed changed when the content is
            // changed.
            const beforeStat = await (0, fs_extra_1.stat)(Setting_1.default.settingFilePath);
            await (0, test_utils_1.msleep)(1001);
            Setting_1.default.setValue('sync.mobileWifiOnly', false);
            await Setting_1.default.saveAll();
            const afterStat = await (0, fs_extra_1.stat)(Setting_1.default.settingFilePath);
            expect(afterStat.mtime.getTime()).toBeGreaterThan(beforeStat.mtime.getTime());
        }
        {
            const beforeStat = await (0, fs_extra_1.stat)(Setting_1.default.settingFilePath);
            await (0, test_utils_1.msleep)(1001);
            Setting_1.default.setValue('sync.mobileWifiOnly', false);
            const afterStat = await (0, fs_extra_1.stat)(Setting_1.default.settingFilePath);
            await Setting_1.default.saveAll();
            expect(afterStat.mtime.getTime()).toBe(beforeStat.mtime.getTime());
        }
    }));
    it('should handle invalid JSON', (async () => {
        const badContent = '{ oopsIforgotTheQuotes: true}';
        await (0, fs_extra_1.writeFile)(Setting_1.default.settingFilePath, badContent, 'utf8');
        await Setting_1.default.reset();
        Logger_1.default.globalLogger.enabled = false;
        await Setting_1.default.load();
        Logger_1.default.globalLogger.enabled = true;
        // Invalid JSON file has been moved to .bak file
        expect(await (0, fs_extra_1.pathExists)(Setting_1.default.settingFilePath)).toBe(false);
        const files = await (0, fs_extra_1.readdir)(Setting_1.default.value('profileDir'));
        expect(files.length).toBe(1);
        expect(files[0].endsWith('.bak')).toBe(true);
        expect(await (0, fs_extra_1.readFile)(`${Setting_1.default.value('profileDir')}/${files[0]}`, 'utf8')).toBe(badContent);
    }));
    it('should allow applying default migrations', (async () => {
        await Setting_1.default.reset();
        expect(Setting_1.default.value('sync.target')).toBe(0);
        expect(Setting_1.default.value('style.editor.contentMaxWidth')).toBe(0);
        await Setting_1.default.applyMigrations();
        expect(Setting_1.default.value('sync.target')).toBe(7); // Changed
        expect(Setting_1.default.value('style.editor.contentMaxWidth')).toBe(600); // Changed
    }));
    it('should skip values that are already set', (async () => {
        await Setting_1.default.reset();
        Setting_1.default.setValue('sync.target', 9);
        await Setting_1.default.applyMigrations();
        expect(Setting_1.default.value('sync.target')).toBe(9); // Not changed
    }));
    it('should allow skipping default migrations', (async () => {
        await Setting_1.default.reset();
        expect(Setting_1.default.value('sync.target')).toBe(0);
        expect(Setting_1.default.value('style.editor.contentMaxWidth')).toBe(0);
        Setting_1.default.skipMigrations();
        await Setting_1.default.applyMigrations();
        expect(Setting_1.default.value('sync.target')).toBe(0); // Not changed
        expect(Setting_1.default.value('style.editor.contentMaxWidth')).toBe(0); // Not changed
    }));
    it('should not apply migrations that have already been applied', (async () => {
        await Setting_1.default.reset();
        Setting_1.default.setValue('lastSettingDefaultMigration', 0);
        expect(Setting_1.default.value('sync.target')).toBe(0);
        expect(Setting_1.default.value('style.editor.contentMaxWidth')).toBe(0);
        await Setting_1.default.applyMigrations();
        expect(Setting_1.default.value('sync.target')).toBe(0); // Not changed
        expect(Setting_1.default.value('style.editor.contentMaxWidth')).toBe(600); // Changed
    }));
    it('should migrate to new setting', (async () => {
        await Setting_1.default.reset();
        Setting_1.default.setValue('spellChecker.language', 'fr-FR');
        await Setting_1.default.applyMigrations();
        expect(Setting_1.default.value('spellChecker.languages')).toStrictEqual(['fr-FR']);
    }));
    it('should not override new setting, if it already set', (async () => {
        await Setting_1.default.reset();
        Setting_1.default.setValue('spellChecker.languages', ['fr-FR', 'en-US']);
        Setting_1.default.setValue('spellChecker.language', 'fr-FR');
        await Setting_1.default.applyMigrations();
        expect(Setting_1.default.value('spellChecker.languages')).toStrictEqual(['fr-FR', 'en-US']);
    }));
    it('should not set new setting, if old setting is not set', (async () => {
        await Setting_1.default.reset();
        expect(Setting_1.default.isSet('spellChecker.language')).toBe(false);
        await Setting_1.default.applyMigrations();
        expect(Setting_1.default.isSet('spellChecker.languages')).toBe(false);
    }));
    it('should migrate global to local settings', (async () => {
        await Setting_1.default.reset();
        // Set an initial value -- should store in the global settings
        const initialLayout = { key: 'test' };
        Setting_1.default.setValue('ui.layout', initialLayout);
        await Setting_1.default.saveAll();
        await switchToSubProfileSettings();
        // The migrations should fetch the previous initial layout from the global settings
        expect(Setting_1.default.value('ui.layout')).toEqual({});
        await Setting_1.default.applyMigrations();
        expect(Setting_1.default.value('ui.layout')).toEqual(initialLayout);
        Setting_1.default.setValue('ui.layout', { key: 'test 2' });
        await Setting_1.default.saveAll();
        // Should not overwrite parent settings
        const globalSettings = await loadDefaultProfileSettings();
        expect(globalSettings['ui.layout']).toEqual(initialLayout);
    }));
    it('migrated settings should still be local even if global->local migrations are skipped', async () => {
        await Setting_1.default.reset();
        const defaultSettingValue = Setting_1.default.value('notes.listRendererId');
        // Set an initial value -- should store in the global settings
        Setting_1.default.setValue('notes.listRendererId', 'global-setting-value');
        await Setting_1.default.saveAll();
        await switchToSubProfileSettings();
        expect(Setting_1.default.value('notes.listRendererId')).toBe(defaultSettingValue);
        // .applyMigrations should not apply skipped migrations
        Setting_1.default.skipMigrations();
        await Setting_1.default.applyMigrations();
        expect(Setting_1.default.value('notes.listRendererId')).toBe(defaultSettingValue);
        // The setting should be local -- the parent setting should not be overwritten
        Setting_1.default.setValue('notes.listRendererId', 'some-other-value');
        await Setting_1.default.saveAll();
        const globalSettings = await loadDefaultProfileSettings();
        expect(globalSettings['notes.listRendererId']).toBe('global-setting-value');
    });
    it('should load sub-profile settings', async () => {
        await Setting_1.default.reset();
        await Setting_1.default.registerSetting('non_builtin', {
            public: true,
            storage: Setting_1.SettingStorage.File,
            isGlobal: true,
            type: Setting_1.SettingItemType.Bool,
            value: false,
        });
        Setting_1.default.setValue('locale', 'fr_FR'); // Global setting
        Setting_1.default.setValue('theme', Setting_1.default.THEME_DARK); // Global setting
        Setting_1.default.setValue('sync.target', 9); // Local setting
        Setting_1.default.setValue('non_builtin', true); // Local setting
        await Setting_1.default.saveAll();
        await switchToSubProfileSettings();
        expect(Setting_1.default.value('locale')).toBe('fr_FR'); // Should come from the root profile
        expect(Setting_1.default.value('theme')).toBe(Setting_1.default.THEME_DARK); // Should come from the root profile
        expect(Setting_1.default.value('sync.target')).toBe(0); // Should come from the local profile
        // Non-built-in variables are not copied
        expect(() => Setting_1.default.value('non_builtin')).toThrow();
        // Also check that the special loadOne() function works as expected
        expect((await Setting_1.default.loadOne('locale')).value).toBe('fr_FR');
        expect((await Setting_1.default.loadOne('theme')).value).toBe(Setting_1.default.THEME_DARK);
        expect((await Setting_1.default.loadOne('sync.target'))).toBe(null);
    });
    it('should save sub-profile settings', async () => {
        await Setting_1.default.reset();
        Setting_1.default.setValue('locale', 'fr_FR'); // Global setting
        Setting_1.default.setValue('theme', Setting_1.default.THEME_DARK); // Global setting
        await Setting_1.default.saveAll();
        await switchToSubProfileSettings();
        Setting_1.default.setValue('locale', 'en_GB'); // Should be saved to global
        Setting_1.default.setValue('sync.target', 8); // Should be saved to local
        await Setting_1.default.saveAll();
        await Setting_1.default.reset();
        await Setting_1.default.load();
        expect(Setting_1.default.value('locale')).toBe('en_GB');
        expect(Setting_1.default.value('theme')).toBe(Setting_1.default.THEME_DARK);
        expect(Setting_1.default.value('sync.target')).toBe(8);
        // Double-check that actual file content is correct
        const globalSettings = await loadDefaultProfileSettings();
        const localSettings = JSON.parse(await (0, fs_extra_1.readFile)(`${Setting_1.default.value('profileDir')}/settings-1.json`, 'utf8'));
        expect(globalSettings).toEqual({
            '$schema': 'https://joplinapp.org/schema/settings.json',
            locale: 'en_GB',
            theme: 2,
        });
        expect(localSettings).toEqual({
            '$schema': 'https://joplinapp.org/schema/settings.json',
            'sync.target': 8,
        });
    });
    it('should not erase settings of parent profile', async () => {
        // When a sub-profile settings are saved, we should ensure that the
        // local settings of the root profiles are not lost.
        // https://github.com/laurent22/joplin/issues/6459
        await Setting_1.default.reset();
        Setting_1.default.setValue('sync.target', 9); // Local setting (Root profile)
        await Setting_1.default.saveAll();
        await switchToSubProfileSettings();
        Setting_1.default.setValue('sync.target', 2); // Local setting (Sub-profile)
        await Setting_1.default.saveAll();
        const globalSettings = await loadDefaultProfileSettings();
        expect(globalSettings['sync.target']).toBe(9);
    });
    it('all global settings should be saved to file', async () => {
        for (const [k, v] of Object.entries(Setting_1.default.metadata())) {
            if (v.isGlobal && v.storage !== Setting_1.SettingStorage.File)
                throw new Error(`Setting "${k}" is global but storage is not "file"`);
        }
    });
    test('values should not be undefined when they are set', async () => {
        Setting_1.default.setValue('locale', undefined);
        expect(Setting_1.default.value('locale')).toBe('');
    });
    test('values should not be undefined when registering a setting', async () => {
        await Setting_1.default.registerSetting('myCustom', {
            public: true,
            value: undefined,
            type: Setting_1.default.TYPE_STRING,
        });
        expect(Setting_1.default.value('myCustom')).toBe('');
    });
    test('should not fail Sqlite UNIQUE constraint when re-registering saved settings', async () => {
        // Re-registering a saved database setting previously caused issues with saving.
        for (let i = 0; i < 2; i++) {
            await Setting_1.default.registerSetting('myCustom', {
                public: true,
                value: `${i}`,
                type: Setting_1.default.TYPE_STRING,
                storage: Setting_1.SettingStorage.Database,
            });
            Setting_1.default.setValue('myCustom', 'test');
            await Setting_1.default.saveAll();
        }
    });
    test('should enforce min and max values for when the setting is already in the cache and when it is not', async () => {
        await Setting_1.default.reset();
        Setting_1.default.setValue('revisionService.ttlDays', 0);
        expect(Setting_1.default.value('revisionService.ttlDays')).toBe(1);
        Setting_1.default.setValue('revisionService.ttlDays', 100000);
        expect(Setting_1.default.value('revisionService.ttlDays')).toBe(99999);
        await Setting_1.default.reset();
        Setting_1.default.setValue('revisionService.ttlDays', 100000);
        expect(Setting_1.default.value('revisionService.ttlDays')).toBe(99999);
        Setting_1.default.setValue('revisionService.ttlDays', 0);
        expect(Setting_1.default.value('revisionService.ttlDays')).toBe(1);
    });
    test('should not fail to save settings that can conflict with uninstalled plugin settings', async () => {
        Setting_1.default.setValue('editor.imageRendering', true);
        expect(Setting_1.default.value('editor.imageRendering')).toBe(true);
        Setting_1.default.setValue('editor.imageRendering', false);
        expect(Setting_1.default.value('editor.imageRendering')).toBe(false);
    });
    test('should adjust settings to avoid conflicts', async () => {
        const testSettingId = 'plugin-plugin.calebjohn.rich-markdown.inlineImages';
        await Setting_1.default.registerSetting(testSettingId, {
            public: true,
            value: false,
            type: Setting_1.SettingItemType.Bool,
            storage: Setting_1.SettingStorage.File,
        });
        Setting_1.default.setValue('editor.imageRendering', true);
        // Setting one conflicting setting should update the other
        Setting_1.default.setValue(testSettingId, true);
        expect(Setting_1.default.value('editor.imageRendering')).toBe(false);
        expect(Setting_1.default.value(testSettingId)).toBe(true);
        Setting_1.default.setValue('editor.imageRendering', true);
        expect(Setting_1.default.value('editor.imageRendering')).toBe(true);
        expect(Setting_1.default.value(testSettingId)).toBe(false);
    });
});
//# sourceMappingURL=Setting.test.js.map