"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const BaseItem_1 = require("../models/BaseItem");
const BaseModel_1 = require("../BaseModel");
const MasterKey_1 = require("../models/MasterKey");
const Resource_1 = require("../models/Resource");
const ResourceService_1 = require("./ResourceService");
const Logger_1 = require("@joplin/utils/Logger");
const shim_1 = require("../shim");
const PerformanceLogger_1 = require("../PerformanceLogger");
const AsyncActionQueue_1 = require("../AsyncActionQueue");
const EventEmitter = require('events');
const perfLogger = PerformanceLogger_1.default.create();
// Key for use with the KvStore.
const decryptionErrorKeyPrefix = 'decryptErrorLabel:';
const decryptionErrorKey = (type, id) => {
    return `${decryptionErrorKeyPrefix}${type}:${id}`;
};
const decryptionCounterKeyPrefix = 'decrypt:';
const decryptionCounterKey = (type, id) => {
    return `${decryptionCounterKeyPrefix}${type}:${id}`;
};
class DecryptionWorker {
    constructor() {
        this.state_ = 'idle';
        // eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
        this.dispatch = () => { };
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        this.scheduleId_ = null;
        this.kvStore_ = null;
        this.maxDecryptionAttempts_ = 2;
        this.taskQueue_ = new AsyncActionQueue_1.default();
        this.encryptionService_ = null;
        this.state_ = 'idle';
        this.logger_ = new Logger_1.default();
        this.eventEmitter_ = new EventEmitter();
    }
    setLogger(l) {
        this.logger_ = l;
    }
    logger() {
        return this.logger_;
    }
    // eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
    on(eventName, callback) {
        return this.eventEmitter_.on(eventName, callback);
    }
    // eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
    off(eventName, callback) {
        return this.eventEmitter_.removeListener(eventName, callback);
    }
    static instance() {
        if (DecryptionWorker.instance_)
            return DecryptionWorker.instance_;
        DecryptionWorker.instance_ = new DecryptionWorker();
        return DecryptionWorker.instance_;
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    setEncryptionService(v) {
        this.encryptionService_ = v;
    }
    setKvStore(v) {
        this.kvStore_ = v;
    }
    encryptionService() {
        if (!this.encryptionService_)
            throw new Error('DecryptionWorker.encryptionService_ is not set!!');
        return this.encryptionService_;
    }
    kvStore() {
        if (!this.kvStore_)
            throw new Error('DecryptionWorker.kvStore_ is not set!!');
        return this.kvStore_;
    }
    async scheduleStart() {
        if (this.scheduleId_)
            return;
        this.scheduleId_ = shim_1.default.setTimeout(() => {
            this.scheduleId_ = null;
            void this.start({
                masterKeyNotLoadedHandler: 'dispatch',
            });
        }, 1000);
    }
    async decryptionDisabledItems() {
        let items = await this.kvStore().searchByPrefix(decryptionCounterKeyPrefix);
        items = items.filter(item => item.value > this.maxDecryptionAttempts_);
        return await Promise.all(items.map(async (item) => {
            const s = item.key.split(':');
            const type_ = Number(s[1]);
            const id = s[2];
            const errorDescription = await this.kvStore().value(decryptionErrorKey(type_, id));
            return {
                type_,
                id,
                reason: errorDescription,
            };
        }));
    }
    async clearDisabledItem(typeId, itemId) {
        await this.kvStore().deleteValue(decryptionCounterKey(typeId, itemId));
        await this.kvStore().deleteValue(decryptionErrorKey(typeId, itemId));
    }
    async clearDisabledItems() {
        await this.kvStore().deleteByPrefix(decryptionCounterKeyPrefix);
        await this.kvStore().deleteByPrefix(decryptionErrorKeyPrefix);
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    dispatchReport(report) {
        const action = Object.assign({}, report);
        action.type = 'DECRYPTION_WORKER_SET';
        this.dispatch(action);
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async start_(options = null) {
        if (options === null)
            options = {};
        if (!('masterKeyNotLoadedHandler' in options))
            options.masterKeyNotLoadedHandler = 'throw';
        if (!('errorHandler' in options))
            options.errorHandler = 'log';
        if (this.state_ !== 'idle') {
            const msg = `DecryptionWorker: cannot start because state is "${this.state_}"`;
            this.logger().debug(msg);
            return { error: new Error(msg) };
        }
        // Note: the logic below is an optimisation to avoid going through the loop if no master key exists
        // or if none is loaded. It means this logic needs to be duplicate a bit what's in the loop, like the
        // "throw" and "dispatch" logic.
        const loadedMasterKeyCount = await this.encryptionService().loadedMasterKeysCount();
        if (!loadedMasterKeyCount) {
            const msg = 'DecryptionWorker: cannot start because no master key is currently loaded.';
            this.logger().info(msg);
            const ids = await MasterKey_1.default.allIds();
            // Note that the current implementation means that a warning will be
            // displayed even if the user has no encrypted note. Just having
            // encrypted master key is sufficient. Not great but good enough for
            // now.
            if (ids.length) {
                if (options.masterKeyNotLoadedHandler === 'throw') {
                    // By trying to load the master key here, we throw the "masterKeyNotLoaded" error
                    // which the caller needs.
                    await this.encryptionService().loadedMasterKey(ids[0]);
                }
                else {
                    this.dispatch({
                        type: 'MASTERKEY_SET_NOT_LOADED',
                        ids: ids,
                    });
                }
            }
            return { error: new Error(msg) };
        }
        this.logger().info('DecryptionWorker: starting decryption...');
        this.state_ = 'started';
        const excludedIds = [];
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        const decryptedItemCounts = {};
        let skippedItemCount = 0;
        this.dispatch({ type: 'ENCRYPTION_HAS_DISABLED_ITEMS', value: false });
        this.dispatchReport({ state: 'started' });
        const decryptItemsTask = perfLogger.taskStart('DecryptionWorker/decryptItems');
        try {
            const notLoadedMasterKeyDispatches = [];
            while (true) {
                const result = await BaseItem_1.default.itemsThatNeedDecryption(excludedIds);
                const items = result.items;
                for (let i = 0; i < items.length; i++) {
                    const item = items[i];
                    const ItemClass = BaseItem_1.default.itemClass(item);
                    this.dispatchReport({
                        itemIndex: i,
                        itemCount: items.length,
                    });
                    const counterKey = decryptionCounterKey(item.type_, item.id);
                    const errorKey = decryptionErrorKey(item.type_, item.id);
                    const clearDecryptionCounter = async () => {
                        await this.kvStore().deleteValue(counterKey);
                        // The decryption error key stores the reason for the decryption counter's value.
                        // As such, the error should be reset when the decryption counter is reset:
                        await this.kvStore().deleteValue(errorKey);
                    };
                    // Don't log in production as it results in many messages when importing many items
                    // this.logger().debug('DecryptionWorker: decrypting: ' + item.id + ' (' + ItemClass.tableName() + ')');
                    try {
                        const decryptCounter = await this.kvStore().incValue(counterKey);
                        if (decryptCounter > this.maxDecryptionAttempts_) {
                            this.logger().debug(`DecryptionWorker: ${BaseModel_1.default.modelTypeToName(item.type_)} ${item.id}: Decryption has failed more than 2 times - skipping it`);
                            this.dispatch({ type: 'ENCRYPTION_HAS_DISABLED_ITEMS', value: true });
                            skippedItemCount++;
                            excludedIds.push(item.id);
                            continue;
                        }
                        const decryptedItem = await ItemClass.decrypt(item);
                        await clearDecryptionCounter();
                        if (!decryptedItemCounts[decryptedItem.type_])
                            decryptedItemCounts[decryptedItem.type_] = 0;
                        decryptedItemCounts[decryptedItem.type_]++;
                        if (decryptedItem.type_ === Resource_1.default.modelType() && !!decryptedItem.encryption_blob_encrypted) {
                            // itemsThatNeedDecryption() will return the resource again if the blob has not been decrypted,
                            // but that will result in an infinite loop if the blob simply has not been downloaded yet.
                            // So skip the ID for now, and the service will try to decrypt the blob again the next time.
                            excludedIds.push(decryptedItem.id);
                        }
                        if (decryptedItem.type_ === Resource_1.default.modelType() && !decryptedItem.encryption_blob_encrypted) {
                            this.eventEmitter_.emit('resourceDecrypted', { id: decryptedItem.id });
                        }
                        if (decryptedItem.type_ === Resource_1.default.modelType() && !decryptedItem.encryption_applied && !!decryptedItem.encryption_blob_encrypted) {
                            this.eventEmitter_.emit('resourceMetadataButNotBlobDecrypted', { id: decryptedItem.id });
                        }
                    }
                    catch (error) {
                        excludedIds.push(item.id);
                        if (error.code === 'masterKeyNotLoaded' && options.masterKeyNotLoadedHandler === 'dispatch') {
                            if (notLoadedMasterKeyDispatches.indexOf(error.masterKeyId) < 0) {
                                this.dispatch({
                                    type: 'MASTERKEY_ADD_NOT_LOADED',
                                    id: error.masterKeyId,
                                });
                                notLoadedMasterKeyDispatches.push(error.masterKeyId);
                            }
                            await clearDecryptionCounter();
                            continue;
                        }
                        if (error.code === 'masterKeyNotLoaded' && options.masterKeyNotLoadedHandler === 'throw') {
                            await clearDecryptionCounter();
                            throw error;
                        }
                        await this.kvStore().setValue(errorKey, String(error));
                        if (options.errorHandler === 'log') {
                            this.logger().warn(`DecryptionWorker: error for: ${item.id} (${ItemClass.tableName()})`, error);
                            this.logger().debug('Item with error:', item);
                        }
                        else {
                            throw error;
                        }
                    }
                }
                if (!result.hasMore)
                    break;
            }
        }
        catch (error) {
            this.logger().error('DecryptionWorker:', error);
            this.state_ = 'idle';
            this.dispatchReport({ state: 'idle' });
            throw error;
        }
        finally {
            decryptItemsTask.onEnd();
        }
        // 2019-05-12: Temporary to set the file size of the resources
        // that weren't set in migration/20.js due to being on the sync target
        await ResourceService_1.default.autoSetFileSizes();
        this.logger().info('DecryptionWorker: completed decryption.');
        const downloadedButEncryptedBlobCount = await Resource_1.default.downloadedButEncryptedBlobCount(excludedIds);
        this.state_ = 'idle';
        let decryptedItemCount = 0;
        for (const itemType in decryptedItemCounts)
            decryptedItemCount += decryptedItemCounts[itemType];
        const finalReport = {
            skippedItemCount: skippedItemCount,
            decryptedItemCounts: decryptedItemCounts,
            decryptedItemCount: decryptedItemCount,
            error: null,
        };
        this.dispatchReport(Object.assign(Object.assign({}, finalReport), { state: 'idle' }));
        if (downloadedButEncryptedBlobCount) {
            this.logger().info(`DecryptionWorker: Some resources have been downloaded but are not decrypted yet. Scheduling another decryption. Resource count: ${downloadedButEncryptedBlobCount}`);
            void this.scheduleStart();
        }
        return finalReport;
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async start(options = {}) {
        let output = null;
        let lastError;
        // Use taskQueue_ to ensure that only one decryption task is running at a time.
        this.taskQueue_.push(async () => {
            const startTask = perfLogger.taskStart('DecryptionWorker/start');
            try {
                output = await this.start_(options);
            }
            catch (error) {
                lastError = error;
            }
            finally {
                startTask.onEnd();
            }
        });
        await this.taskQueue_.processAllNow();
        if (lastError) {
            throw lastError;
        }
        return output;
    }
    async destroy() {
        this.eventEmitter_.removeAllListeners();
        if (this.scheduleId_) {
            shim_1.default.clearTimeout(this.scheduleId_);
            this.scheduleId_ = null;
        }
        this.eventEmitter_ = null;
        DecryptionWorker.instance_ = null;
        await this.taskQueue_.waitForAllDone();
    }
}
DecryptionWorker.instance_ = null;
exports.default = DecryptionWorker;
//# sourceMappingURL=DecryptionWorker.js.map