"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const buffer_1 = require("buffer");
const cryptoShared_1 = require("./cryptoShared");
const pbkdf2Raw = async (password, salt, iterations, keylenBytes, digest) => {
    const encoder = new TextEncoder();
    const key = await crypto_1.webcrypto.subtle.importKey('raw', encoder.encode(password), { name: 'PBKDF2' }, false, ['deriveKey']);
    return crypto_1.webcrypto.subtle.deriveKey({ name: 'PBKDF2', salt, iterations, hash: digest }, key, { name: 'AES-GCM', length: keylenBytes * 8 }, false, ['encrypt', 'decrypt']);
};
const encryptRaw = async (data, key, iv, authTagLengthBytes, additionalData) => {
    return buffer_1.Buffer.from(await crypto_1.webcrypto.subtle.encrypt({
        name: 'AES-GCM',
        iv,
        additionalData,
        tagLength: authTagLengthBytes * 8,
    }, key, data));
};
const decryptRaw = async (data, key, iv, authTagLengthBytes, associatedData) => {
    return buffer_1.Buffer.from(await crypto_1.webcrypto.subtle.decrypt({
        name: 'AES-GCM',
        iv,
        additionalData: associatedData,
        tagLength: authTagLengthBytes * 8,
    }, key, data));
};
const crypto = {
    randomBytes: async (size) => {
        // .getRandomValues has a maximum output size
        const maxChunkSize = 65536;
        const result = new Uint8Array(size);
        if (size <= maxChunkSize) {
            crypto_1.webcrypto.getRandomValues(result);
        }
        else {
            const fullSizeChunk = new Uint8Array(maxChunkSize);
            const lastChunkSize = size % maxChunkSize;
            const maxOffset = size - lastChunkSize;
            let offset = 0;
            while (offset < maxOffset) {
                crypto_1.webcrypto.getRandomValues(fullSizeChunk);
                result.set(fullSizeChunk, offset);
                offset += maxChunkSize;
            }
            if (lastChunkSize > 0) {
                const lastChunk = crypto_1.webcrypto.getRandomValues(new Uint8Array(lastChunkSize));
                result.set(lastChunk, offset);
            }
        }
        return buffer_1.Buffer.from(result);
    },
    digest: async (algorithm, data) => {
        return buffer_1.Buffer.from(await crypto_1.webcrypto.subtle.digest(algorithm, data));
    },
    encrypt: async (password, salt, data, encryptionParameters) => {
        // Parameters in EncryptionParameters won't appear in result
        const result = {
            salt: salt.toString('base64'),
            iv: '',
            ct: '', // cipherText
        };
        // 96 bits IV
        // "For IVs, it is recommended that implementations restrict support to the length of 96 bits, to promote interoperability, efficiency, and simplicity of design." - NIST SP 800-38D
        const iv = await crypto.randomBytes(12);
        const key = await pbkdf2Raw(password, salt, encryptionParameters.iterationCount, encryptionParameters.keyLength, encryptionParameters.digestAlgorithm);
        const encrypted = await encryptRaw(data, key, iv, encryptionParameters.authTagLength, encryptionParameters.associatedData);
        result.iv = iv.toString('base64');
        result.ct = encrypted.toString('base64');
        return result;
    },
    decrypt: async (password, data, encryptionParameters) => {
        const salt = buffer_1.Buffer.from(data.salt, 'base64');
        const iv = buffer_1.Buffer.from(data.iv, 'base64');
        const key = await pbkdf2Raw(password, salt, encryptionParameters.iterationCount, encryptionParameters.keyLength, encryptionParameters.digestAlgorithm);
        const decrypted = decryptRaw(buffer_1.Buffer.from(data.ct, 'base64'), key, iv, encryptionParameters.authTagLength, encryptionParameters.associatedData);
        return decrypted;
    },
    encryptString: async (password, salt, data, encoding, encryptionParameters) => {
        return crypto.encrypt(password, salt, buffer_1.Buffer.from(data, encoding), encryptionParameters);
    },
    generateNonce: cryptoShared_1.generateNonce,
    increaseNonce: cryptoShared_1.increaseNonce,
};
(0, cryptoShared_1.setRandomBytesImplementation)(crypto.randomBytes);
exports.default = crypto;
//# sourceMappingURL=crypto.js.map