"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const KeychainServiceDriverBase_1 = require("./KeychainServiceDriverBase");
const shim_1 = require("../../shim");
const Logger_1 = require("@joplin/utils/Logger");
const KvStore_1 = require("../KvStore");
const Setting_1 = require("../../models/Setting");
const logger = Logger_1.default.create('KeychainServiceDriver.electron');
const canUseSafeStorage = () => {
    var _a, _b, _c;
    return !!((_c = (_b = (_a = shim_1.default.electronBridge) === null || _a === void 0 ? void 0 : _a.call(shim_1.default)) === null || _b === void 0 ? void 0 : _b.safeStorage) === null || _c === void 0 ? void 0 : _c.isEncryptionAvailable());
};
const kvStorePrefix = 'KeychainServiceDriver.secureStore.';
class KeychainServiceDriver extends KeychainServiceDriverBase_1.default {
    constructor(appId, clientId) {
        var _a, _b;
        super(appId, clientId);
        this.driverId = 'electron-safeStorage';
        if (canUseSafeStorage() && shim_1.default.isLinux()) {
            logger.info('KeychainService Linux backend: ', (_b = (_a = shim_1.default.electronBridge()) === null || _a === void 0 ? void 0 : _a.safeStorage) === null || _b === void 0 ? void 0 : _b.getSelectedStorageBackend());
        }
    }
    async supported() {
        return canUseSafeStorage();
    }
    async setPassword(name, password) {
        if (canUseSafeStorage()) {
            logger.debug('Saving password with electron safe storage. ID: ', name);
            let encrypted;
            try {
                encrypted = await shim_1.default.electronBridge().safeStorage.encryptString(password);
            }
            catch (error) {
                logger.warn('Encrypting a setting failed. Missing keychain permission?', error);
                return false;
            }
            await KvStore_1.default.instance().setValue(`${kvStorePrefix}${name}`, encrypted);
        }
        else {
            // Unsupported.
            return false;
        }
        return true;
    }
    async password(name) {
        let result = null;
        if (canUseSafeStorage()) {
            const data = await KvStore_1.default.instance().value(`${kvStorePrefix}${name}`);
            if (data !== null) {
                try {
                    result = await shim_1.default.electronBridge().safeStorage.decryptString(data);
                }
                catch (e) {
                    logger.warn('Decryption of a setting failed. Corrupted data or new keychain password? Error:', e);
                    if (shim_1.default.isLinux() && Setting_1.default.value('env') === 'dev') {
                        logger.warn('If running Joplin in development mode with NodeJS installed from the Snap store, consider retrying with NodeJS installed from a different source.');
                    }
                }
            }
        }
        return result;
    }
    async deletePassword(name) {
        if (canUseSafeStorage()) {
            logger.debug('Trying to delete encrypted password with id ', name);
            await KvStore_1.default.instance().deleteValue(`${kvStorePrefix}${name}`);
        }
    }
}
exports.default = KeychainServiceDriver;
//# sourceMappingURL=KeychainServiceDriver.electron.js.map