"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Setting_1 = require("../../models/Setting");
const test_utils_synchronizer_1 = require("../../testing/test-utils-synchronizer");
const test_utils_1 = require("../../testing/test-utils");
const Folder_1 = require("../../models/Folder");
const Note_1 = require("../../models/Note");
const BaseItem_1 = require("../../models/BaseItem");
const WelcomeUtils_1 = require("../../WelcomeUtils");
const syncInfoUtils_1 = require("./syncInfoUtils");
const errors_1 = require("../../errors");
describe('Synchronizer.basics', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(2);
        await (0, test_utils_1.switchClient)(1);
        (0, test_utils_1.synchronizer)().testingHooks_ = [];
    });
    afterAll(async () => {
        await (0, test_utils_1.afterAllCleanUp)();
    });
    it('should create remote items', (async () => {
        const folder = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'un', parent_id: folder.id });
        const all = await (0, test_utils_synchronizer_1.allNotesFolders)();
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_synchronizer_1.localNotesFoldersSameAsRemote)(all, expect);
    }));
    it('should update remote items', (async () => {
        const folder = await Folder_1.default.save({ title: 'folder1' });
        const note = await Note_1.default.save({ title: 'un', parent_id: folder.id });
        await (0, test_utils_1.synchronizerStart)();
        await Note_1.default.save({ title: 'un UPDATE', id: note.id });
        const all = await (0, test_utils_synchronizer_1.allNotesFolders)();
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_synchronizer_1.localNotesFoldersSameAsRemote)(all, expect);
    }));
    it('should create local items', (async () => {
        const folder = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'un', parent_id: folder.id });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        const all = await (0, test_utils_synchronizer_1.allNotesFolders)();
        await (0, test_utils_synchronizer_1.localNotesFoldersSameAsRemote)(all, expect);
    }));
    it('should update local items', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'un', parent_id: folder1.id });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.sleep)(0.1);
        let note2 = await Note_1.default.load(note1.id);
        note2.title = 'Updated on client 2';
        await Note_1.default.save(note2);
        note2 = await Note_1.default.load(note2.id);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizerStart)();
        const all = await (0, test_utils_synchronizer_1.allNotesFolders)();
        await (0, test_utils_synchronizer_1.localNotesFoldersSameAsRemote)(all, expect);
    }));
    it('should delete remote notes', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'un', parent_id: folder1.id });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.sleep)(0.1);
        await Note_1.default.delete(note1.id);
        await (0, test_utils_1.synchronizerStart)();
        const remotes = await (0, test_utils_synchronizer_1.remoteNotesAndFolders)();
        expect(remotes.length).toBe(1);
        expect(remotes[0].id).toBe(folder1.id);
        const deletedItems = await BaseItem_1.default.deletedItems((0, test_utils_1.syncTargetId)());
        expect(deletedItems.length).toBe(0);
    }));
    it('should not created deleted_items entries for items deleted via sync', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'un', parent_id: folder1.id });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await Folder_1.default.delete(folder1.id);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizerStart)();
        const deletedItems = await BaseItem_1.default.deletedItems((0, test_utils_1.syncTargetId)());
        expect(deletedItems.length).toBe(0);
    }));
    it('should delete local notes', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'un', parent_id: folder1.id });
        const note2 = await Note_1.default.save({ title: 'deux', parent_id: folder1.id });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await Note_1.default.delete(note1.id);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizerStart)();
        const items = await (0, test_utils_synchronizer_1.allNotesFolders)();
        expect(items.length).toBe(2);
        const deletedItems = await BaseItem_1.default.deletedItems((0, test_utils_1.syncTargetId)());
        expect(deletedItems.length).toBe(0);
        await Note_1.default.delete(note2.id);
        await (0, test_utils_1.synchronizerStart)();
    }));
    it('should delete remote folder', (async () => {
        await Folder_1.default.save({ title: 'folder1' });
        const folder2 = await Folder_1.default.save({ title: 'folder2' });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.sleep)(0.1);
        await Folder_1.default.delete(folder2.id);
        await (0, test_utils_1.synchronizerStart)();
        const all = await (0, test_utils_synchronizer_1.allNotesFolders)();
        await (0, test_utils_synchronizer_1.localNotesFoldersSameAsRemote)(all, expect);
    }));
    it('should delete local folder', (async () => {
        await Folder_1.default.save({ title: 'folder1' });
        const folder2 = await Folder_1.default.save({ title: 'folder2' });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await Folder_1.default.delete(folder2.id);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizerStart)();
        const items = await (0, test_utils_synchronizer_1.allNotesFolders)();
        await (0, test_utils_synchronizer_1.localNotesFoldersSameAsRemote)(items, expect);
    }));
    it('should cross delete all folders', (async () => {
        // If client1 and 2 have two folders, client 1 deletes item 1 and client
        // 2 deletes item 2, they should both end up with no items after sync.
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const folder2 = await Folder_1.default.save({ title: 'folder2' });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.sleep)(0.1);
        await Folder_1.default.delete(folder1.id);
        await (0, test_utils_1.switchClient)(1);
        await Folder_1.default.delete(folder2.id);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        const items2 = await (0, test_utils_synchronizer_1.allNotesFolders)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizerStart)();
        const items1 = await (0, test_utils_synchronizer_1.allNotesFolders)();
        expect(items1.length).toBe(0);
        expect(items1.length).toBe(items2.length);
    }));
    it('items should be downloaded again when user cancels in the middle of delta operation', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'un', is_todo: 1, parent_id: folder1.id });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        (0, test_utils_1.synchronizer)().testingHooks_ = ['cancelDeltaLoop2'];
        await (0, test_utils_1.synchronizerStart)();
        let notes = await Note_1.default.all();
        expect(notes.length).toBe(0);
        (0, test_utils_1.synchronizer)().testingHooks_ = [];
        await (0, test_utils_1.synchronizerStart)();
        notes = await Note_1.default.all();
        expect(notes.length).toBe(1);
    }));
    it('should skip items that cannot be synced', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'un', is_todo: 1, parent_id: folder1.id });
        const noteId = note1.id;
        await (0, test_utils_1.synchronizerStart)();
        let disabledItems = await BaseItem_1.default.syncDisabledItems((0, test_utils_1.syncTargetId)());
        expect(disabledItems.length).toBe(0);
        await Note_1.default.save({ id: noteId, title: 'un mod' });
        (0, test_utils_1.synchronizer)().testingHooks_ = ['notesRejectedByTarget'];
        await (0, test_utils_1.synchronizerStart)();
        (0, test_utils_1.synchronizer)().testingHooks_ = [];
        await (0, test_utils_1.synchronizerStart)(); // Another sync to check that this item is now excluded from sync
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        const notes = await Note_1.default.all();
        expect(notes.length).toBe(1);
        expect(notes[0].title).toBe('un');
        await (0, test_utils_1.switchClient)(1);
        disabledItems = await BaseItem_1.default.syncDisabledItems((0, test_utils_1.syncTargetId)());
        expect(disabledItems.length).toBe(1);
    }));
    it('should handle items that are read-only on the sync target', (async () => {
        const folder = await Folder_1.default.save({ title: 'folder' });
        const note = await Note_1.default.save({ title: 'un', is_todo: 1, parent_id: folder.id });
        const noteId = note.id;
        await (0, test_utils_1.synchronizerStart)();
        await Note_1.default.save({ id: noteId, title: 'un mod' });
        (0, test_utils_1.synchronizer)().testingHooks_ = ['itemIsReadOnly'];
        await (0, test_utils_1.synchronizerStart)();
        (0, test_utils_1.synchronizer)().testingHooks_ = [];
        const noteReload = await Note_1.default.load(note.id);
        expect(noteReload.title).toBe(note.title);
        const conflictNote = (await Note_1.default.all()).find((n) => !!n.is_conflict);
        expect(conflictNote).toBeTruthy();
        expect(conflictNote.title).toBe('un mod');
        expect(conflictNote.id).not.toBe(note.id);
    }));
    it('should allow duplicate folder titles', (async () => {
        await Folder_1.default.save({ title: 'folder' });
        await (0, test_utils_1.switchClient)(2);
        let remoteF2 = await Folder_1.default.save({ title: 'folder' });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.sleep)(0.1);
        await (0, test_utils_1.synchronizerStart)();
        const localF2 = await Folder_1.default.load(remoteF2.id);
        expect(localF2.title === remoteF2.title).toBe(true);
        // Then that folder that has been renamed locally should be set in such a way
        // that synchronizing it applies the title change remotely, and that new title
        // should be retrieved by client 2.
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.sleep)(0.1);
        await (0, test_utils_1.synchronizerStart)();
        remoteF2 = await Folder_1.default.load(remoteF2.id);
        expect(remoteF2.title === localF2.title).toBe(true);
    }));
    it('should create remote items with UTF-8 content', (async () => {
        const folder = await Folder_1.default.save({ title: 'Fahrräder' });
        await Note_1.default.save({ title: 'Fahrräder', body: 'Fahrräder', parent_id: folder.id });
        const all = await (0, test_utils_synchronizer_1.allNotesFolders)();
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_synchronizer_1.localNotesFoldersSameAsRemote)(all, expect);
    }));
    it('should update remote items but not pull remote changes', (async () => {
        const folder = await Folder_1.default.save({ title: 'folder1' });
        const note = await Note_1.default.save({ title: 'un', parent_id: folder.id });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await Note_1.default.save({ title: 'deux', parent_id: folder.id });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await Note_1.default.save({ title: 'un UPDATE', id: note.id });
        await (0, test_utils_1.synchronizerStart)(null, { syncSteps: ['update_remote'] });
        const all = await (0, test_utils_synchronizer_1.allNotesFolders)();
        expect(all.length).toBe(2);
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        const note2 = await Note_1.default.load(note.id);
        expect(note2.title).toBe('un UPDATE');
    }));
    it('should create a new Welcome notebook on each client', (async () => {
        // Create the Welcome items on two separate clients
        await WelcomeUtils_1.default.createWelcomeItems('en_GB');
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await WelcomeUtils_1.default.createWelcomeItems('en_GB');
        const beforeFolderCount = (await Folder_1.default.all()).length;
        const beforeNoteCount = (await Note_1.default.all()).length;
        expect(beforeFolderCount === 1).toBe(true);
        expect(beforeNoteCount > 1).toBe(true);
        await (0, test_utils_1.synchronizerStart)();
        const afterFolderCount = (await Folder_1.default.all()).length;
        const afterNoteCount = (await Note_1.default.all()).length;
        expect(afterFolderCount).toBe(beforeFolderCount * 2);
        expect(afterNoteCount).toBe(beforeNoteCount * 2);
        // Changes to the Welcome items should be synced to all clients
        const f1 = (await Folder_1.default.all())[0];
        await Folder_1.default.save({ id: f1.id, title: 'Welcome MOD' });
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizerStart)();
        const f1_1 = await Folder_1.default.load(f1.id);
        expect(f1_1.title).toBe('Welcome MOD');
    }));
    it('should not wipe out user data when syncing with an empty target', (async () => {
        // Only these targets support the wipeOutFailSafe flag (in other words, the targets that use basicDelta)
        if (!['nextcloud', 'memory', 'filesystem', 'amazon_s3'].includes((0, test_utils_1.syncTargetName)()))
            return;
        for (let i = 0; i < 10; i++)
            await Note_1.default.save({ title: 'note' });
        Setting_1.default.setValue('sync.wipeOutFailSafe', true);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.fileApi)().clearRoot(); // oops
        await (0, test_utils_1.synchronizerStart)();
        expect((await Note_1.default.all()).length).toBe(10); // but since the fail-safe if on, the notes have not been deleted
        Setting_1.default.setValue('sync.wipeOutFailSafe', false); // Now switch it off
        await (0, test_utils_1.synchronizerStart)();
        expect((await Note_1.default.all()).length).toBe(0); // Since the fail-safe was off, the data has been cleared
        // Handle case where the sync target has been wiped out, then the user creates one note and sync.
        for (let i = 0; i < 10; i++)
            await Note_1.default.save({ title: 'note' });
        Setting_1.default.setValue('sync.wipeOutFailSafe', true);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.fileApi)().clearRoot();
        await Note_1.default.save({ title: 'ma note encore' });
        await (0, test_utils_1.synchronizerStart)();
        expect((await Note_1.default.all()).length).toBe(11);
    }));
    it('should not sync deletions that came via sync even when there is a conflict', (async () => {
        // This test is mainly to simulate sharing, unsharing and sharing a note
        // again. Previously, when doing so, the app would create deleted_items
        // objects on the recipient when the owner unshares. It means that when
        // sharing again, the recipient would apply the deletions and delete
        // everything in the shared notebook.
        //
        // Specifically it was happening when a conflict was generated as a
        // result of the items being deleted.
        //
        // - C1 creates a note and sync
        // - C2 sync and get the note
        // - C2 deletes the note and sync
        // - C1 modify the note, and sync
        //
        // => A conflict is created. The note is deleted and a copy is created
        // in the Conflict folder.
        //
        // After this, we recreate the note on the sync target (simulates the
        // note being shared again), and we check that C2 doesn't attempt to
        // delete that note.
        const note = await Note_1.default.save({});
        await (0, test_utils_1.synchronizerStart)();
        const noteSerialized = await (0, test_utils_1.fileApi)().get(`${note.id}.md`);
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await Note_1.default.delete(note.id);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await Note_1.default.save({ id: note.id });
        await (0, test_utils_1.synchronizerStart)();
        expect((await Note_1.default.all())[0].is_conflict).toBe(1);
        await (0, test_utils_1.fileApi)().put(`${note.id}.md`, noteSerialized); // Recreate the note - simulate sharing again.
        await (0, test_utils_1.synchronizerStart)();
        // Check that the client didn't delete the note
        const remotes = (await (0, test_utils_1.fileApi)().list()).items;
        expect(remotes.find(r => r.path === `${note.id}.md`)).toBeTruthy();
    }));
    it('should throw an error if the app version is not compatible with the sync target info', (async () => {
        await (0, test_utils_1.synchronizerStart)();
        const remoteInfo = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.synchronizer)().api());
        remoteInfo.appMinVersion = '100.0.0';
        await (0, syncInfoUtils_1.uploadSyncInfo)((0, test_utils_1.synchronizer)().api(), remoteInfo);
        await (0, test_utils_1.expectThrow)(async () => (0, test_utils_1.synchronizerStart)(1, {
            throwOnError: true,
        }), errors_1.ErrorCode.MustUpgradeApp);
    }));
    it('should update the remote appMinVersion when synchronising', (async () => {
        await (0, test_utils_1.synchronizerStart)();
        const remoteInfoBefore = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.synchronizer)().api());
        // Simulates upgrading the client
        (0, syncInfoUtils_1.setAppMinVersion)('100.0.0');
        await (0, test_utils_1.synchronizerStart)();
        // Then after sync, appMinVersion should be the same as that client version
        const remoteInfoAfter = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.synchronizer)().api());
        expect(remoteInfoBefore.appMinVersion).toBe('3.0.0');
        expect(remoteInfoAfter.appMinVersion).toBe('100.0.0');
        // Now simulates synchronising with an older client version. In that case, it should not be
        // allowed and the remote info.json should not change.
        (0, syncInfoUtils_1.setAppMinVersion)('80.0.0');
        await (0, test_utils_1.expectThrow)(async () => (0, test_utils_1.synchronizerStart)(1, { throwOnError: true }), errors_1.ErrorCode.MustUpgradeApp);
        expect((await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.synchronizer)().api())).appMinVersion).toBe('100.0.0');
    }));
});
//# sourceMappingURL=Synchronizer.basics.test.js.map