"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const test_utils_1 = require("../../testing/test-utils");
const MasterKey_1 = require("../../models/MasterKey");
const syncInfoUtils_1 = require("./syncInfoUtils");
const Setting_1 = require("../../models/Setting");
const BaseItem_1 = require("../../models/BaseItem");
const BaseItem_2 = require("../../models/BaseItem");
const Logger_1 = require("@joplin/utils/Logger");
describe('syncInfoUtils', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.fileApi)().clearRoot();
    });
    afterAll(async () => {
        await (0, test_utils_1.afterAllCleanUp)();
    });
    it('should enable or disable a master key', async () => {
        const mk1 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey('111111'));
        const mk2 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey('111111'));
        (0, syncInfoUtils_1.setMasterKeyEnabled)(mk2.id, false);
        expect((0, syncInfoUtils_1.masterKeyEnabled)(await MasterKey_1.default.load(mk1.id))).toBe(true);
        expect((0, syncInfoUtils_1.masterKeyEnabled)(await MasterKey_1.default.load(mk2.id))).toBe(false);
        (0, syncInfoUtils_1.setMasterKeyEnabled)(mk1.id, false);
        expect((0, syncInfoUtils_1.masterKeyEnabled)(await MasterKey_1.default.load(mk1.id))).toBe(false);
        expect((0, syncInfoUtils_1.masterKeyEnabled)(await MasterKey_1.default.load(mk2.id))).toBe(false);
        (0, syncInfoUtils_1.setMasterKeyEnabled)(mk1.id, true);
        expect((0, syncInfoUtils_1.masterKeyEnabled)(await MasterKey_1.default.load(mk1.id))).toBe(true);
        expect((0, syncInfoUtils_1.masterKeyEnabled)(await MasterKey_1.default.load(mk2.id))).toBe(false);
    });
    it('should tell if two sync info are equal', async () => {
        {
            const syncInfo1 = new syncInfoUtils_1.SyncInfo();
            const syncInfo2 = new syncInfoUtils_1.SyncInfo();
            expect((0, syncInfoUtils_1.syncInfoEquals)(syncInfo1, syncInfo2)).toBe(true);
        }
        {
            const syncInfo1 = new syncInfoUtils_1.SyncInfo();
            syncInfo1.masterKeys = [{
                    id: 'id',
                    content: 'content',
                }];
            const syncInfo2 = new syncInfoUtils_1.SyncInfo();
            syncInfo2.masterKeys = [{
                    id: 'id',
                    content: 'different',
                }];
            expect((0, syncInfoUtils_1.syncInfoEquals)(syncInfo1, syncInfo2)).toBe(false);
        }
        {
            const syncInfo1 = new syncInfoUtils_1.SyncInfo();
            syncInfo1.masterKeys = [{
                    id: 'id',
                    content: 'content',
                }];
            const syncInfo2 = new syncInfoUtils_1.SyncInfo();
            syncInfo2.masterKeys = [{
                    id: 'id',
                    content: 'content',
                }];
            expect((0, syncInfoUtils_1.syncInfoEquals)(syncInfo1, syncInfo2)).toBe(true);
        }
        {
            // Should disregard object key order
            const syncInfo1 = new syncInfoUtils_1.SyncInfo();
            syncInfo1.masterKeys = [{
                    content: 'content',
                    id: 'id',
                }];
            const syncInfo2 = new syncInfoUtils_1.SyncInfo();
            syncInfo2.masterKeys = [{
                    id: 'id',
                    content: 'content',
                }];
            expect((0, syncInfoUtils_1.syncInfoEquals)(syncInfo1, syncInfo2)).toBe(true);
        }
    });
    it('should merge sync target info and keep the highest appMinVersion', async () => {
        const syncInfo1 = new syncInfoUtils_1.SyncInfo();
        syncInfo1.appMinVersion = '1.0.5';
        const syncInfo2 = new syncInfoUtils_1.SyncInfo();
        syncInfo2.appMinVersion = '1.0.2';
        expect((0, syncInfoUtils_1.mergeSyncInfos)(syncInfo1, syncInfo2).appMinVersion).toBe('1.0.5');
        syncInfo1.appMinVersion = '2.1.0';
        syncInfo2.appMinVersion = '2.2.5';
        expect((0, syncInfoUtils_1.mergeSyncInfos)(syncInfo1, syncInfo2).appMinVersion).toBe('2.2.5');
        syncInfo1.appMinVersion = '1.0.0';
        syncInfo2.appMinVersion = '1.0.0';
        expect((0, syncInfoUtils_1.mergeSyncInfos)(syncInfo1, syncInfo2).appMinVersion).toBe('1.0.0');
        // Should prefer the version from syncInfo1 if versions are otherwise equal.
        syncInfo1.appMinVersion = '1.00';
        syncInfo2.appMinVersion = '1.0.0';
        expect((0, syncInfoUtils_1.mergeSyncInfos)(syncInfo1, syncInfo2).appMinVersion).toBe('1.00');
        syncInfo1.appMinVersion = '0.0.0';
        syncInfo2.appMinVersion = '0.00';
        expect((0, syncInfoUtils_1.mergeSyncInfos)(syncInfo1, syncInfo2).appMinVersion).toBe('0.0.0');
    });
    it('should merge sync target info and takes into account usage of master key - 1', async () => {
        const syncInfo1 = new syncInfoUtils_1.SyncInfo();
        syncInfo1.masterKeys = [{
                id: '1',
                content: 'content1',
                hasBeenUsed: true,
            }];
        syncInfo1.activeMasterKeyId = '1';
        await (0, test_utils_1.msleep)(1);
        const syncInfo2 = new syncInfoUtils_1.SyncInfo();
        syncInfo2.masterKeys = [{
                id: '2',
                content: 'content2',
                hasBeenUsed: false,
            }];
        syncInfo2.activeMasterKeyId = '2';
        // If one master key has been used and the other not, it should select
        // the one that's been used regardless of timestamps.
        expect((0, syncInfoUtils_1.mergeSyncInfos)(syncInfo1, syncInfo2).activeMasterKeyId).toBe('1');
        // If both master keys have been used it should rely on timestamp
        // (latest modified is picked).
        syncInfo2.masterKeys[0].hasBeenUsed = true;
        expect((0, syncInfoUtils_1.mergeSyncInfos)(syncInfo1, syncInfo2).activeMasterKeyId).toBe('2');
    });
    it('should merge sync target info, but should not make a disabled key the active one', async () => {
        const syncInfo1 = new syncInfoUtils_1.SyncInfo();
        syncInfo1.masterKeys = [{
                id: '1',
                content: 'content1',
                hasBeenUsed: true,
                enabled: 0,
            }];
        syncInfo1.activeMasterKeyId = '1';
        await (0, test_utils_1.msleep)(1);
        const syncInfo2 = new syncInfoUtils_1.SyncInfo();
        syncInfo2.masterKeys = [{
                id: '2',
                content: 'content2',
                enabled: 1,
                hasBeenUsed: false,
            }];
        syncInfo2.activeMasterKeyId = '2';
        // Normally, if one master key has been used (1) and the other not (2),
        // it should select the one that's been used regardless of timestamps.
        // **However**, if the key 1 has been disabled by user, it should
        // **not** be picked as the active one. Instead it should use key 2,
        // because it's still enabled.
        expect((0, syncInfoUtils_1.mergeSyncInfos)(syncInfo1, syncInfo2).activeMasterKeyId).toBe('2');
        // If both key are disabled, we go back to the original logic, where we
        // select the key that's been used.
        syncInfo2.masterKeys[0].enabled = 0;
        expect((0, syncInfoUtils_1.mergeSyncInfos)(syncInfo1, syncInfo2).activeMasterKeyId).toBe('1');
    });
    it('should fix the sync info if it contains invalid data', async () => {
        test_utils_1.logger.enabled = false;
        const syncInfo = new syncInfoUtils_1.SyncInfo();
        syncInfo.masterKeys = [{
                id: '1',
                content: 'content1',
                hasBeenUsed: true,
                enabled: 0,
            }];
        syncInfo.activeMasterKeyId = '2';
        (0, syncInfoUtils_1.saveLocalSyncInfo)(syncInfo);
        const loaded = (0, syncInfoUtils_1.localSyncInfo)();
        expect(loaded.activeMasterKeyId).toBe('');
        expect(loaded.masterKeys.length).toBe(1);
        test_utils_1.logger.enabled = true;
    });
    // cSpell:disable
    it('should filter unnecessary sync info', async () => {
        const initialData = {
            'version': 3,
            'e2ee': {
                'value': true,
                'updatedTime': 0,
            },
            'activeMasterKeyId': {
                'value': '400227d2222c4d3bb7346514861c643b',
                'updatedTime': 0,
            },
            'masterKeys': [
                {
                    'id': '400227d8a77c4d3bb7346514861c643b',
                    'created_time': 1515008161362,
                    'updated_time': 1708103706234,
                    'source_application': 'net.cozic.joplin-desktop',
                    'encryption_method': 4,
                    'checksum': '',
                    'content': '{"iv":"M1uezlW1Pu1g3dwrCTqcHg==","v":1,"iter":10000,"ks":256,"ts":64,"mode":"ccm","adata":"","cipher":"aes","salt":"0dqWvU/PUVQ=","ct":"wHXN5pk1s7qKX+2Y9puEGZGkojI1Pvc+TvZUKC6QCfwxtMK6C1Hmgvm53vAaeCMcCXPvGVLo9JwqINFhEgb0ux+KUFcCqgT1pNO2Sf/hJsH8PjaUvl0kwpC511zdnvY7Hk3WIpgXVKUevsQt9TkMK5e8y1JMsuuTD3fW7bEiv/ehe4CBSQ9eH1tWjr1qQ=="}',
                    'hasBeenUsed': true,
                },
            ],
            'ppk': {
                'value': {
                    'id': 'SNQ5ZCs61KDVUW2qqqqHd3',
                    'keySize': 2048,
                    'privateKey': {
                        'encryptionMethod': 4,
                        'ciphertext': '{"iv":"Z2y11b4nCYvpmQ9gvxELug==","v":1,"iter":10000,"ks":256,"ts":64,"mode":"ccm","adata":"","cipher":"aes","salt":"0dqWvU/PUVQ=","ct":"8CvjYayXMpLsrAMwtu18liRfewKfZVpRlC0D0I2FYziyFhRf4Cjqi2+Uy8kIC8au7oBSBUnNU6jd04ooNozneKv2MzkhbGlXo3izxqCMVHboqa2vkPWbBAxGlvUYQUg213xG61FjZ19ZJdpti+AQy7qpQU7/h5kyC0iJ2aXG5TIGcBuDq3lbGAVfG/RlL/abMKLYb8KouFYAJe+0bUajUXa1KJsey+eD+2hFVc+nAxKOLe1UoZysB77Lq43DRTBFGH2gTSC1zOXxuZeSbFPPN0Cj+FvV7D5pF9LhUSLPDsIiRwF/q+E506YgDjirSZAvW1Y2EEM22F2Mh0I0pbVPFXhhBafqPLRwXmUGULCnA64gkGwctK5mEs985VVSrpQ0nMvf/drg2vUQrJ3exgl43ddVSOCjeJuF7F06IBL5FQ34iAujsOheRNvlWtG9xm008Vc19NxvhtzIl1RO7XLXrrTBzbFHDrcHjda/xNWNEKwU/LZrH0xPgwEcwBmLItvy/NojI/JKNeck8R431QWooFb7cTplO4qsgCQNL9MJ9avpmNSXJAUQx8VnifKVbzcY4T7X7TmJWSrpvBWV8MLfi3TOF4kahR75vg47kCrMbthFMw5bvrjvMmGOtyKxheqbS5IlSnSSz5x7wIVz0g3vzMbcbb5rF5MuzNhU97wNiz3L1Aonjmnu8r3vCyXTB/4GSiwYH7KfixwYM68T4crqJ0VneNy+owznCdJQXnG4cmjxek1wmJMEmurQ1JtANNv/m43gzoqd62V6Dq05vLJF+n7CS9HgJ3FTqYVCZLGGYrSilIYnEjhdaBpkcnFrCitbfYj+IpNC6eN6qg2hpGAbmKId7RLOGwJyda0jkuNP9mTqWOF+6eYn8Q+Y3YIY"}',
                    },
                    'publicKey': '-----BEGIN RSA PUBLIC KEY-----\nMIIBCgKCAQEAiSTY5wBscae/WmU3PfVP5FYQiuTi5V7BjPcge/6pXvgF3zwe43uy\nTWdzO2YgK/a8f3H507clcGlZN4e0e1jZ/rh4lMfaN\nugfNo0RAvuwn8Yniqfb69reygJywbFBIauxbBpVKbc21MLuCbPkVFjKG7qGNYdF4\nc17mQ8nQsbFPZcuvxsZvgvvbza1q0rqVETdDUClyIrY8plAjMgTKCRwq2gafP6eX\nWpkENAyIbOFxSKXjWy0yFidvZfYLz4mIRwIDAQAB\n-----END RSA PUBLIC KEY-----',
                    'createdTime': 1633274368892,
                },
                'updatedTime': 1633274368892,
            },
            'appMinVersion': '0.0.0',
        };
        const syncInfo = new syncInfoUtils_1.SyncInfo();
        syncInfo.load(JSON.stringify(initialData));
        const filteredSyncInfo = syncInfo.filterSyncInfo();
        expect(filteredSyncInfo).toEqual({
            'activeMasterKeyId': {
                'updatedTime': 0,
                'value': '400227d2222c4d3bb7346514861c643b',
            },
            'appMinVersion': '0.0.0',
            'e2ee': {
                'updatedTime': 0,
                'value': true,
            },
            'masterKeys': [
                {
                    'created_time': 1515008161362,
                    'encryption_method': 4,
                    'hasBeenUsed': true,
                    'id': '400227d8a77c4d3bb7346514861c643b',
                    'source_application': 'net.cozic.joplin-desktop',
                    'updated_time': 1708103706234,
                },
            ],
            'ppk': {
                'updatedTime': 1633274368892,
                'value': {
                    'createdTime': 1633274368892,
                    'id': 'SNQ5ZCs61KDVUW2qqqqHd3',
                    'keySize': 2048,
                    'privateKey': {
                        'ciphertext': '{"iv":"Z2y11b4nCYvpm...TqWOF+6eYn8Q+Y3YIY"}',
                        'encryptionMethod': 4,
                    },
                    'publicKey': '-----BEGIN RSA PUBLIC KEY-----\nMIIBCgKCA...',
                },
            },
            'version': 3,
        });
    });
    // cSpell:enable
    test.each([
        ['1.0.0', '1.0.4', true],
        ['1.0.0', '0.0.5', false],
        ['1.0.0', '1.0.0', true],
    ])('should check if it can sync', async (appMinVersion, appVersion, expected) => {
        let succeeded = true;
        try {
            const s = new syncInfoUtils_1.SyncInfo();
            s.appMinVersion = appMinVersion;
            (0, syncInfoUtils_1.checkIfCanSync)(s, appVersion);
        }
        catch (error) {
            succeeded = false;
        }
        expect(succeeded).toBe(expected);
    });
    test('should not throw if the sync info being parsed is invalid', async () => {
        Logger_1.default.globalLogger.enabled = false;
        Setting_1.default.setValue('syncInfoCache', 'invalid-json');
        expect(() => (0, syncInfoUtils_1.localSyncInfo)()).not.toThrow();
        Logger_1.default.globalLogger.enabled = true;
    });
    test('should use default value if the sync info being parsed is invalid', async () => {
        Logger_1.default.globalLogger.enabled = false;
        Setting_1.default.setValue('syncInfoCache', 'invalid-json');
        const result = (0, syncInfoUtils_1.localSyncInfo)();
        expect(result.activeMasterKeyId).toEqual('');
        expect(result.version).toEqual(0);
        expect(result.ppk).toEqual(null);
        expect(result.e2ee).toEqual(false);
        expect(result.appMinVersion).toEqual('3.0.0');
        expect(result.masterKeys).toEqual([]);
        Logger_1.default.globalLogger.enabled = true;
    });
    it('should succeed when info.json exists for checkSyncTargetIsValid', (async () => {
        Setting_1.default.setValue('sync.wipeOutFailSafe', true);
        const syncInfo = new syncInfoUtils_1.SyncInfo();
        await (0, test_utils_1.fileApi)().put('info.json', syncInfo.serialize());
        expect((0, syncInfoUtils_1.checkSyncTargetIsValid)((0, test_utils_1.fileApi)())).resolves.not.toThrow();
    }));
    it('should succeed when info.json does not exist and failsafe is disabled for checkSyncTargetIsValid', (async () => {
        Setting_1.default.setValue('sync.wipeOutFailSafe', false);
        expect((0, syncInfoUtils_1.checkSyncTargetIsValid)((0, test_utils_1.fileApi)())).resolves.not.toThrow();
    }));
    it('should fail with failsafe error when info.json does not exist for checkSyncTargetIsValid', (async () => {
        Setting_1.default.setValue('sync.wipeOutFailSafe', true);
        await expect((0, syncInfoUtils_1.checkSyncTargetIsValid)((0, test_utils_1.fileApi)())).rejects.toThrow('Fail-safe: ');
    }));
    it('should succeed when info.json exists and is valid for fetchSyncInfo', (async () => {
        Setting_1.default.setValue('sync.wipeOutFailSafe', true);
        const expectedSyncInfo = new syncInfoUtils_1.SyncInfo();
        expectedSyncInfo.version = 50;
        await (0, test_utils_1.fileApi)().put('info.json', expectedSyncInfo.serialize());
        const actualSyncInfo = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)());
        expect(actualSyncInfo).toStrictEqual(expectedSyncInfo);
    }));
    it('should fail with missing version error when info.json exists but is invalid for fetchSyncInfo', (async () => {
        Setting_1.default.setValue('sync.wipeOutFailSafe', true);
        await (0, test_utils_1.fileApi)().put('info.json', new syncInfoUtils_1.SyncInfo().serialize());
        await expect((0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)())).rejects.toThrow('Missing "version" field');
    }));
    it('should succeed when info.json does not exist but .sync/version.txt does exist for fetchSyncInfo', (async () => {
        Setting_1.default.setValue('sync.wipeOutFailSafe', true);
        await (0, test_utils_1.fileApi)().put('.sync/version.txt', '{}');
        const actualSyncInfo = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)());
        expect(actualSyncInfo.version).toBe(1);
    }));
    it('should succeed when info.json and .sync/version.txt does not exist and failsafe is disabled for fetchSyncInfo', (async () => {
        Setting_1.default.setValue('sync.wipeOutFailSafe', false);
        const actualSyncInfo = await (0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)());
        expect(actualSyncInfo.version).toBe(0);
    }));
    it('should fail with failsafe error when info.json and .sync/version.txt does not exist when sync items are present for fetchSyncInfo', (async () => {
        Setting_1.default.setValue('sync.wipeOutFailSafe', true);
        const note = {
            id: 1,
            type_: BaseItem_2.default.TYPE_NOTE,
        };
        await BaseItem_1.default.saveSyncTime((0, test_utils_1.fileApi)().syncTargetId(), note, 1);
        await expect((0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)())).rejects.toThrow('Fail-safe: ');
    }));
    it('should succeed when info.json and .sync/version.txt does not exist when sync items are not present for fetchSyncInfo', (async () => {
        Setting_1.default.setValue('sync.wipeOutFailSafe', true);
        expect((0, syncInfoUtils_1.fetchSyncInfo)((0, test_utils_1.fileApi)())).resolves.not.toThrow();
    }));
});
//# sourceMappingURL=syncInfoUtils.test.js.map