"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs_extra_1 = require("fs-extra");
const test_utils_1 = require("./testing/test-utils");
const RotatingLogs_1 = require("./RotatingLogs");
const createTestLogFile = async (dir) => {
    await (0, fs_extra_1.writeFile)(`${dir}/log.txt`, 'some content');
};
describe('RotatingLogs', () => {
    test('should rename log.txt to log-TIMESTAMP.txt', async () => {
        let dir;
        try {
            dir = await (0, test_utils_1.createTempDir)();
            await createTestLogFile(dir);
            let files = await (0, fs_extra_1.readdir)(dir);
            expect(files.find(file => file.match(/^log.txt$/gi))).toBeTruthy();
            expect(files.length).toBe(1);
            const rotatingLogs = new RotatingLogs_1.default(dir, 1, 1);
            await rotatingLogs.cleanActiveLogFile();
            files = await (0, fs_extra_1.readdir)(dir);
            expect(files.find(file => file.match(/^log.txt$/gi))).toBeFalsy();
            expect(files.find(file => file.match(/^log-[0-9]+.txt$/gi))).toBeTruthy();
            expect(files.length).toBe(1);
        }
        finally {
            await (0, fs_extra_1.remove)(dir);
        }
    });
    test('should delete inactive log file after 1ms', async () => {
        let dir;
        try {
            dir = await (0, test_utils_1.createTempDir)();
            await createTestLogFile(dir);
            const rotatingLogs = new RotatingLogs_1.default(dir, 1, 1);
            await rotatingLogs.cleanActiveLogFile();
            await (0, test_utils_1.msleep)(1);
            await rotatingLogs.deleteNonActiveLogFiles();
            const files = await (0, fs_extra_1.readdir)(dir);
            expect(files.find(file => file.match(/^log-[0-9]+.txt$/gi))).toBeFalsy();
            expect(files.length).toBe(0);
        }
        finally {
            await (0, fs_extra_1.remove)(dir);
        }
    });
    test('should not delete the log-timestamp.txt right after its be created', async () => {
        let dir;
        try {
            dir = await (0, test_utils_1.createTempDir)();
            await createTestLogFile(dir);
            const rotatingLogs = new RotatingLogs_1.default(dir, 1, 5000);
            await rotatingLogs.cleanActiveLogFile();
            await rotatingLogs.deleteNonActiveLogFiles();
            const files = await (0, fs_extra_1.readdir)(dir);
            expect(files.find(file => file.match(/^log-[0-9]+.txt$/gi))).toBeTruthy();
            expect(files.length).toBe(1);
        }
        finally {
            await (0, fs_extra_1.remove)(dir);
        }
    });
});
//# sourceMappingURL=RotatingLogs.test.js.map