"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const redux_1 = require("redux");
const CommandService_1 = require("../CommandService");
const ToolbarButtonUtils_1 = require("./ToolbarButtonUtils");
const reducer_1 = require("../../reducer");
const stateToWhenClauseContext_1 = require("./stateToWhenClauseContext");
const createTestCommands = () => {
    const simpleCommands = [
        { name: 'testCommand1', label: 'Test 1' },
        { name: 'testCommand2', label: 'Test 2' },
    ].map(({ name, label }) => ({
        declaration: {
            name,
            label: () => label,
            iconName: 'fa-times',
        },
        runtime: () => ({
            execute: () => Promise.resolve(),
        }),
    }));
    const commands = [
        ...simpleCommands,
        {
            declaration: {
                name: 'invisibleUnlessTrashSelected',
                label: () => 'Invisible',
            },
            runtime: () => ({
                execute: () => Promise.resolve(),
                visibleCondition: 'inTrash',
            }),
        },
    ];
    return commands;
};
describe('ToolbarButtonUtils', () => {
    beforeAll(() => {
        const store = (0, redux_1.createStore)(reducer_1.default);
        CommandService_1.default.instance().initialize(store, false, stateToWhenClauseContext_1.default);
        const commands = createTestCommands();
        for (const command of commands) {
            CommandService_1.default.instance().registerDeclaration(command.declaration);
        }
        CommandService_1.default.instance().registerCommands(commands);
    });
    test('should convert command names to toolbar buttons', () => {
        const utils = new ToolbarButtonUtils_1.default(CommandService_1.default.instance());
        const buttons = utils.commandsToToolbarButtons(['testCommand1', 'testCommand2'], (0, stateToWhenClauseContext_1.default)(reducer_1.defaultState));
        expect(buttons).toMatchObject([
            {
                type: 'button',
                name: 'testCommand1',
                tooltip: 'Test 1',
                enabled: true,
            },
            {
                type: 'button',
                name: 'testCommand2',
                tooltip: 'Test 2',
                enabled: true,
            },
        ]);
    });
    test('should not repeat separators', () => {
        const utils = new ToolbarButtonUtils_1.default(CommandService_1.default.instance());
        const buttons = utils.commandsToToolbarButtons(['testCommand2', '-', '-', '-', 'testCommand1'], (0, stateToWhenClauseContext_1.default)(reducer_1.defaultState));
        expect(buttons).toMatchObject([
            {
                type: 'button',
                name: 'testCommand2',
            },
            { type: 'separator' },
            {
                type: 'button',
                name: 'testCommand1',
            },
        ]);
    });
    test('should not show invisible commands', () => {
        const utils = new ToolbarButtonUtils_1.default(CommandService_1.default.instance());
        const defaultContext = (0, stateToWhenClauseContext_1.default)(reducer_1.defaultState);
        expect(utils.commandsToToolbarButtons(['invisibleUnlessTrashSelected'], defaultContext)).toMatchObject([]);
        expect(utils.commandsToToolbarButtons(['invisibleUnlessTrashSelected'], Object.assign(Object.assign({}, defaultContext), { inTrash: true }))).toMatchObject([
            { type: 'button', name: 'invisibleUnlessTrashSelected' },
        ]);
    });
});
//# sourceMappingURL=ToolbarButtonUtils.test.js.map