"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const reducer_1 = require("../../reducer");
const SyncTargetRegistry_1 = require("../../SyncTargetRegistry");
const trash_1 = require("../trash");
const stateToWhenClauseContext_1 = require("./stateToWhenClauseContext");
const buildState = (options) => {
    return Object.assign(Object.assign(Object.assign({}, reducer_1.defaultState), { selectedFolderIds: options.selectedFolderId ? [options.selectedFolderId] : [] }), options);
};
describe('stateToWhenClauseContext', () => {
    it('should be in trash if selected note has been deleted and selected folder is trash', async () => {
        const applicationState = buildState({
            selectedNoteIds: ['1'],
            selectedFolderId: (0, trash_1.getTrashFolderId)(),
            notes: [
                { id: '1', deleted_time: 1722567036580 },
            ],
            folders: [],
        });
        const resultingState = (0, stateToWhenClauseContext_1.default)(applicationState);
        expect(resultingState.inTrash).toBe(true);
    });
    it('should NOT be in trash if selected note has not been deleted', async () => {
        const applicationState = buildState({
            selectedNoteIds: ['1'],
            selectedFolderId: (0, trash_1.getTrashFolderId)(),
            notes: [
                { id: '1', deleted_time: 0 },
            ],
            folders: [],
        });
        const resultingState = (0, stateToWhenClauseContext_1.default)(applicationState);
        expect(resultingState.inTrash).toBe(false);
    });
    it('should NOT be in trash if selected folder is not trash', async () => {
        const applicationState = buildState({
            selectedNoteIds: ['1'],
            selectedFolderId: 'any-other-folder',
            notes: [
                { id: '1', deleted_time: 1722567036580 },
            ],
            folders: [],
        });
        const resultingState = (0, stateToWhenClauseContext_1.default)(applicationState);
        expect(resultingState.inTrash).toBe(false);
    });
    it('should be in trash if command folder is deleted', async () => {
        const applicationState = buildState({
            notes: [],
            notesParentType: 'Folder',
            folders: [
                { id: '1', deleted_time: 1722567036580, share_id: '', parent_id: '' },
            ],
        });
        const resultingState = (0, stateToWhenClauseContext_1.default)(applicationState, { commandFolderId: '1' });
        expect(resultingState.inTrash).toBe(true);
    });
    it('should NOT be in trash if command folder is not deleted', async () => {
        const applicationState = buildState({
            notes: [],
            folders: [
                { id: '1', deleted_time: 0, share_id: '', parent_id: '' },
            ],
        });
        const resultingState = (0, stateToWhenClauseContext_1.default)(applicationState, { commandFolderId: '1' });
        expect(resultingState.inTrash).toBe(false);
    });
    it('should not be in trash if viewing all notes', async () => {
        const applicationState = buildState({
            selectedFolderId: 'folder',
            notesParentType: 'SmartFolder',
        });
        const resultingState = (0, stateToWhenClauseContext_1.default)(applicationState);
        expect(resultingState.inTrash).toBe(false);
    });
    it.each(SyncTargetRegistry_1.default.allIds().map(id => ({
        id,
        name: SyncTargetRegistry_1.default.idToName(id),
        expected: SyncTargetRegistry_1.default.isJoplinServerOrCloud(id),
    })))('should set joplinServerConnected to $expected when the sync target is $name', ({ id, expected }) => {
        const getWhenClauseContext = (syncTarget) => {
            const applicationState = Object.assign(Object.assign({}, reducer_1.defaultState), { settings: {
                    'sync.target': syncTarget,
                } });
            return (0, stateToWhenClauseContext_1.default)(applicationState);
        };
        const whenClauseContext = getWhenClauseContext(id);
        expect(whenClauseContext.joplinServerConnected).toBe(expected);
    });
    it.each([
        {
            label: 'should be true when all target folders are deleted',
            commandFolderIds: ['del1', 'del2'],
            expectedDeletedState: true,
        },
        {
            label: 'should be false when one or more target folders is not deleted',
            commandFolderIds: ['del1', 'del2', '1'],
            expectedDeletedState: false,
        },
        {
            label: 'should default to the state of the selected items',
            commandFolderIds: null,
            expectedDeletedState: false,
        },
    ])('should set foldersAreDeleted correctly: $label', ({ commandFolderIds, expectedDeletedState }) => {
        const applicationState = buildState({
            folders: [
                { id: '1', deleted_time: 0, share_id: '', parent_id: '' },
                { id: 'del1', deleted_time: 1, share_id: '', parent_id: '' },
                { id: 'del2', deleted_time: 1, share_id: '', parent_id: '' },
            ],
            selectedFolderIds: ['1'],
            selectedFolderId: '1',
            notesParentType: 'Folder',
        });
        expect((0, stateToWhenClauseContext_1.default)(applicationState, { commandFolderIds })).toHaveProperty('foldersAreDeleted', expectedDeletedState);
    });
});
//# sourceMappingURL=stateToWhenClauseContext.test.js.map