"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const test_utils_1 = require("../../testing/test-utils");
const Folder_1 = require("../../models/Folder");
const Note_1 = require("../../models/Note");
const Setting_1 = require("../../models/Setting");
const BaseItem_1 = require("../../models/BaseItem");
const MasterKey_1 = require("../../models/MasterKey");
const EncryptionService_1 = require("./EncryptionService");
const syncInfoUtils_1 = require("../synchronizer/syncInfoUtils");
let service = null;
describe('services_EncryptionService', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        service = new EncryptionService_1.default();
        BaseItem_1.default.encryptionService_ = service;
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
    });
    it('should encode and decode header', (async () => {
        const header = {
            encryptionMethod: EncryptionService_1.EncryptionMethod.SJCL,
            masterKeyId: '01234568abcdefgh01234568abcdefgh',
        };
        const encodedHeader = service.encodeHeader_(header);
        const decodedHeader = service.decodeHeaderBytes_(encodedHeader);
        delete decodedHeader.length;
        expect((0, test_utils_1.objectsEqual)(header, decodedHeader)).toBe(true);
    }));
    it('should generate and decrypt a master key', (async () => {
        const masterKey = await service.generateMasterKey('123456');
        expect(!!masterKey.content).toBe(true);
        let hasThrown = false;
        try {
            await service.decryptMasterKeyContent(masterKey, 'wrongpassword');
        }
        catch (error) {
            hasThrown = true;
        }
        expect(hasThrown).toBe(true);
        const decryptedMasterKey = await service.decryptMasterKeyContent(masterKey, '123456');
        expect(decryptedMasterKey.length).toBe(512);
    }));
    it('should upgrade a master key', (async () => {
        // Create an old style master key
        let masterKey = await service.generateMasterKey('123456', {
            encryptionMethod: EncryptionService_1.EncryptionMethod.SJCL2,
        });
        masterKey = await MasterKey_1.default.save(masterKey);
        let upgradedMasterKey = await service.reencryptMasterKey(masterKey, '123456', '123456');
        upgradedMasterKey = await MasterKey_1.default.save(upgradedMasterKey);
        // Check that master key has been upgraded (different ciphertext)
        expect(masterKey.content).not.toBe(upgradedMasterKey.content);
        // Check that master key plain text is still the same
        const plainTextOld = await service.decryptMasterKeyContent(masterKey, '123456');
        const plainTextNew = await service.decryptMasterKeyContent(upgradedMasterKey, '123456');
        expect(plainTextOld).toBe(plainTextNew);
        // Check that old content can be decrypted with new master key
        await service.loadMasterKey(masterKey, '123456', true);
        const cipherText = await service.encryptString('some secret');
        const plainTextFromOld = await service.decryptString(cipherText);
        await service.loadMasterKey(upgradedMasterKey, '123456', true);
        const plainTextFromNew = await service.decryptString(cipherText);
        expect(plainTextFromOld).toBe(plainTextFromNew);
    }));
    it('should not upgrade master key if invalid password', (async () => {
        const masterKey = await service.generateMasterKey('123456', {
            encryptionMethod: EncryptionService_1.EncryptionMethod.SJCL2,
        });
        await (0, test_utils_1.checkThrowAsync)(async () => await service.reencryptMasterKey(masterKey, '777', '777'));
    }));
    it('should require a checksum only for old master keys', (async () => {
        const masterKey = await service.generateMasterKey('123456', {
            encryptionMethod: EncryptionService_1.EncryptionMethod.SJCL2,
        });
        expect(!!masterKey.checksum).toBe(true);
        expect(!!masterKey.content).toBe(true);
    }));
    it.each([
        EncryptionService_1.EncryptionMethod.SJCL4,
        EncryptionService_1.EncryptionMethod.KeyV1,
    ])('should not require a checksum for new master keys', (async (masterKeyEncryptionMethod) => {
        const masterKey = await service.generateMasterKey('123456', {
            encryptionMethod: masterKeyEncryptionMethod,
        });
        expect(!masterKey.checksum).toBe(true);
        expect(!!masterKey.content).toBe(true);
        const decryptedMasterKey = await service.decryptMasterKeyContent(masterKey, '123456');
        expect(decryptedMasterKey.length).toBe(512);
    }));
    it.each([
        EncryptionService_1.EncryptionMethod.SJCL4,
        EncryptionService_1.EncryptionMethod.KeyV1,
    ])('should throw an error if master key decryption fails', (async (masterKeyEncryptionMethod) => {
        const masterKey = await service.generateMasterKey('123456', {
            encryptionMethod: masterKeyEncryptionMethod,
        });
        const hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await service.decryptMasterKeyContent(masterKey, 'wrong'));
        expect(hasThrown).toBe(true);
    }));
    it('should return the master keys that need an upgrade', (async () => {
        const masterKey1 = await MasterKey_1.default.save(await service.generateMasterKey('123456', {
            encryptionMethod: EncryptionService_1.EncryptionMethod.SJCL2,
        }));
        const masterKey2 = await MasterKey_1.default.save(await service.generateMasterKey('123456', {
            encryptionMethod: EncryptionService_1.EncryptionMethod.SJCL,
        }));
        await MasterKey_1.default.save(await service.generateMasterKey('123456'));
        const needUpgrade = service.masterKeysThatNeedUpgrading(await MasterKey_1.default.all());
        expect(needUpgrade.length).toBe(2);
        expect(needUpgrade.map(k => k.id).sort()).toEqual([masterKey1.id, masterKey2.id].sort());
    }));
    it('should encrypt and decrypt with a master key', (async () => {
        let masterKey = await service.generateMasterKey('123456');
        masterKey = await MasterKey_1.default.save(masterKey);
        await service.loadMasterKey(masterKey, '123456', true);
        const cipherText = await service.encryptString('some secret');
        const plainText = await service.decryptString(cipherText);
        expect(plainText).toBe('some secret');
        // Test that a long string, that is going to be split into multiple chunks, encrypt
        // and decrypt properly too.
        let veryLongSecret = '';
        for (let i = 0; i < service.chunkSize(service.defaultEncryptionMethod()) * 3; i++)
            veryLongSecret += Math.floor(Math.random() * 9);
        const cipherText2 = await service.encryptString(veryLongSecret);
        const plainText2 = await service.decryptString(cipherText2);
        expect(plainText2 === veryLongSecret).toBe(true);
    }));
    it('should decrypt various encryption methods', (async () => {
        let masterKey = await service.generateMasterKey('123456');
        masterKey = await MasterKey_1.default.save(masterKey);
        await service.loadMasterKey(masterKey, '123456', true);
        {
            const cipherText = await service.encryptString('some secret', {
                encryptionMethod: EncryptionService_1.EncryptionMethod.SJCL2,
            });
            const plainText = await service.decryptString(cipherText);
            expect(plainText).toBe('some secret');
            const header = await service.decodeHeaderString(cipherText);
            expect(header.encryptionMethod).toBe(EncryptionService_1.EncryptionMethod.SJCL2);
        }
        {
            const cipherText = await service.encryptString('some secret', {
                encryptionMethod: EncryptionService_1.EncryptionMethod.SJCL3,
            });
            const plainText = await service.decryptString(cipherText);
            expect(plainText).toBe('some secret');
            const header = await service.decodeHeaderString(cipherText);
            expect(header.encryptionMethod).toBe(EncryptionService_1.EncryptionMethod.SJCL3);
        }
    }));
    it('should fail to decrypt if master key not present', (async () => {
        let masterKey = await service.generateMasterKey('123456');
        masterKey = await MasterKey_1.default.save(masterKey);
        await service.loadMasterKey(masterKey, '123456', true);
        const cipherText = await service.encryptString('some secret');
        await service.unloadMasterKey(masterKey);
        const hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await service.decryptString(cipherText));
        expect(hasThrown).toBe(true);
    }));
    it('should fail to decrypt if data tampered with', (async () => {
        let masterKey = await service.generateMasterKey('123456');
        masterKey = await MasterKey_1.default.save(masterKey);
        await service.loadMasterKey(masterKey, '123456', true);
        let cipherText = await service.encryptString('some secret');
        cipherText += 'ABCDEFGHIJ';
        const hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await service.decryptString(cipherText));
        expect(hasThrown).toBe(true);
    }));
    it.each([
        EncryptionService_1.EncryptionMethod.SJCL1a,
        EncryptionService_1.EncryptionMethod.SJCL1b,
        EncryptionService_1.EncryptionMethod.SJCL4,
        EncryptionService_1.EncryptionMethod.KeyV1,
        EncryptionService_1.EncryptionMethod.FileV1,
        EncryptionService_1.EncryptionMethod.StringV1,
    ])('should fail to decrypt if ciphertext is not a valid JSON string', (async (jsonCipherTextMethod) => {
        const masterKey = await service.generateMasterKey('123456');
        const masterKeyContent = await service.decryptMasterKeyContent(masterKey, '123456');
        const cipherTextString = await service.encrypt(jsonCipherTextMethod, masterKeyContent, 'e21de21d'); // 'e21de21d' is a valid base64/hex string
        // Check if decryption is working
        const plainText = await service.decrypt(jsonCipherTextMethod, masterKeyContent, cipherTextString);
        expect(plainText).toBe('e21de21d');
        // Make invalid JSON
        const invalidCipherText = cipherTextString.replace('{', '{,');
        const hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await service.decrypt(jsonCipherTextMethod, masterKeyContent, invalidCipherText));
        expect(hasThrown).toBe(true);
    }));
    it.each([
        EncryptionService_1.EncryptionMethod.SJCL1a,
        EncryptionService_1.EncryptionMethod.SJCL1b,
        EncryptionService_1.EncryptionMethod.SJCL4,
        EncryptionService_1.EncryptionMethod.KeyV1,
        EncryptionService_1.EncryptionMethod.FileV1,
        EncryptionService_1.EncryptionMethod.StringV1,
    ])('should fail to decrypt if ciphertext authentication failed', (async (authenticatedEncryptionMethod) => {
        const masterKey = await service.generateMasterKey('123456');
        const masterKeyContent = await service.decryptMasterKeyContent(masterKey, '123456');
        const cipherTextObject = JSON.parse(await service.encrypt(authenticatedEncryptionMethod, masterKeyContent, 'e21de21d')); // 'e21de21d' is a valid base64/hex string
        expect(cipherTextObject).toHaveProperty('ct');
        const ct = Buffer.from(cipherTextObject['ct'], 'base64');
        // Should not fail if the binary data of ct is not modified
        const oldCipherTextObject = Object.assign(Object.assign({}, cipherTextObject), { ct: ct.toString('base64') });
        const plainText = await service.decrypt(authenticatedEncryptionMethod, masterKeyContent, JSON.stringify(oldCipherTextObject));
        expect(plainText).toBe('e21de21d');
        // The encrypted data part is changed so it doesn't match the authentication tag. Decryption should fail.
        ct[0] ^= 0x55;
        const newCipherTextObject = Object.assign(Object.assign({}, cipherTextObject), { ct: ct.toString('base64') });
        const hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => service.decrypt(authenticatedEncryptionMethod, masterKeyContent, JSON.stringify(newCipherTextObject)));
        expect(hasThrown).toBe(true);
    }));
    it('should encrypt and decrypt notes and folders', (async () => {
        let masterKey = await service.generateMasterKey('123456');
        masterKey = await MasterKey_1.default.save(masterKey);
        await service.loadMasterKey(masterKey, '123456', true);
        const folder = await Folder_1.default.save({ title: 'folder' });
        const note = await Note_1.default.save({ title: 'encrypted note', body: 'something', parent_id: folder.id });
        const serialized = await Note_1.default.serializeForSync(note);
        const deserialized = Note_1.default.filter(await Note_1.default.unserialize(serialized));
        // Check that required properties are not encrypted
        expect(deserialized.id).toBe(note.id);
        expect(deserialized.parent_id).toBe(note.parent_id);
        expect(deserialized.updated_time).toBe(note.updated_time);
        // Check that at least title and body are encrypted
        expect(!deserialized.title).toBe(true);
        expect(!deserialized.body).toBe(true);
        // Check that encrypted data is there
        expect(!!deserialized.encryption_cipher_text).toBe(true);
        const encryptedNote = await Note_1.default.save(deserialized);
        const decryptedNote = await Note_1.default.decrypt(encryptedNote);
        expect(decryptedNote.title).toBe(note.title);
        expect(decryptedNote.body).toBe(note.body);
        expect(decryptedNote.id).toBe(note.id);
        expect(decryptedNote.parent_id).toBe(note.parent_id);
    }));
    it.each([
        EncryptionService_1.EncryptionMethod.SJCL1a,
        EncryptionService_1.EncryptionMethod.SJCL1b,
        EncryptionService_1.EncryptionMethod.FileV1,
        EncryptionService_1.EncryptionMethod.StringV1,
    ])('should encrypt and decrypt files', (async (fileEncryptionMethod) => {
        let masterKey = await service.generateMasterKey('123456');
        masterKey = await MasterKey_1.default.save(masterKey);
        await service.loadMasterKey(masterKey, '123456', true);
        const sourcePath = `${test_utils_1.supportDir}/photo.jpg`;
        const encryptedPath = `${Setting_1.default.value('tempDir')}/photo.crypted`;
        const decryptedPath = `${Setting_1.default.value('tempDir')}/photo.jpg`;
        service.defaultFileEncryptionMethod_ = fileEncryptionMethod;
        await service.encryptFile(sourcePath, encryptedPath);
        await service.decryptFile(encryptedPath, decryptedPath);
        expect((0, test_utils_1.fileContentEqual)(sourcePath, encryptedPath)).toBe(false);
        expect((0, test_utils_1.fileContentEqual)(sourcePath, decryptedPath)).toBe(true);
    }));
    it.each([
        EncryptionService_1.EncryptionMethod.SJCL1a,
        EncryptionService_1.EncryptionMethod.SJCL1b,
        EncryptionService_1.EncryptionMethod.StringV1,
    ])('should encrypt invalid UTF-8 data', (async (stringEncryptionMethod) => {
        let masterKey = await service.generateMasterKey('123456');
        masterKey = await MasterKey_1.default.save(masterKey);
        await service.loadMasterKey(masterKey, '123456', true);
        // First check that we can replicate the error with the old encryption method
        service.defaultEncryptionMethod_ = EncryptionService_1.EncryptionMethod.SJCL;
        const hasThrown = await (0, test_utils_1.checkThrowAsync)(async () => await service.encryptString('🐶🐶🐶'.substr(0, 5)));
        expect(hasThrown).toBe(true);
        // Now check that the new one fixes the problem
        service.defaultEncryptionMethod_ = stringEncryptionMethod;
        const cipherText = await service.encryptString('🐶🐶🐶'.substr(0, 5));
        const plainText = await service.decryptString(cipherText);
        expect(plainText).toBe('🐶🐶🐶'.substr(0, 5));
    }));
    it('should check if a master key is loaded', (async () => {
        let masterKey = await service.generateMasterKey('123456');
        masterKey = await MasterKey_1.default.save(masterKey);
        await service.loadMasterKey(masterKey, '123456', true);
        expect(service.isMasterKeyLoaded(masterKey)).toBe(true);
        await (0, test_utils_1.msleep)(1);
        // If the master key is modified afterwards it should report that it is
        // *not* loaded since it doesn't have this new version.
        masterKey = await MasterKey_1.default.save(masterKey);
        expect(service.isMasterKeyLoaded(masterKey)).toBe(false);
    }));
});
//# sourceMappingURL=EncryptionService.test.js.map