"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.testData = void 0;
exports.createTestData = createTestData;
exports.checkTestData = checkTestData;
exports.deploySyncTargetSnapshot = deploySyncTargetSnapshot;
exports.main = main;
const test_utils_1 = require("../testing/test-utils");
const Setting_1 = require("../models/Setting");
const Folder_1 = require("../models/Folder");
const Note_1 = require("../models/Note");
const Tag_1 = require("../models/Tag");
const Resource_1 = require("../models/Resource");
const markdownUtils_1 = require("../markdownUtils");
const shim_1 = require("../shim");
const fs = require("fs-extra");
const syncInfoUtils_1 = require("../services/synchronizer/syncInfoUtils");
const { shimInit } = require('../shim-init-node');
const sharp = require('sharp');
const nodeSqlite = require('sqlite3');
const snapshotBaseDir = `${test_utils_1.supportDir}/syncTargetSnapshots`;
exports.testData = {
    folder1: {
        subFolder1: {},
        subFolder2: {
            note1: {
                resource: true,
                tags: ['tag1'],
            },
            note2: {},
        },
        note3: {
            tags: ['tag1', 'tag2'],
        },
        note4: {
            tags: ['tag2'],
        },
    },
    folder2: {},
    folder3: {
        note5: {
            resource: true,
            tags: ['tag2'],
        },
    },
};
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
async function createTestData(data) {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async function recurseStruct(s, parentId = '') {
        for (const n in s) {
            if (n.toLowerCase().includes('folder')) {
                const folder = await Folder_1.default.save({ title: n, parent_id: parentId });
                await recurseStruct(s[n], folder.id);
            }
            else {
                const note = await Note_1.default.save({ title: n, parent_id: parentId });
                if (s[n].resource) {
                    await shim_1.default.attachFileToNote(note, `${test_utils_1.supportDir}/photo.jpg`);
                }
                if (s[n].tags) {
                    for (const tagTitle of s[n].tags) {
                        await Tag_1.default.addNoteTagByTitle(note.id, tagTitle);
                    }
                }
            }
        }
    }
    await recurseStruct(data);
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
async function checkTestData(data) {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async function recurseCheck(s) {
        for (const n in s) {
            const obj = s[n];
            if (n.toLowerCase().includes('folder')) {
                const folder = await Folder_1.default.loadByTitle(n);
                if (!folder)
                    throw new Error(`Cannot load folder: ${n}`);
                await recurseCheck(obj);
            }
            else {
                const note = await Note_1.default.loadByTitle(n);
                if (!note)
                    throw new Error(`Cannot load note: ${n}`);
                const parent = await Folder_1.default.load(note.parent_id);
                if (!parent)
                    throw new Error(`Cannot load note parent: ${n}`);
                if (obj.resource) {
                    const urls = markdownUtils_1.default.extractImageUrls(note.body);
                    const resourceId = urls[0].substr(2);
                    const resource = await Resource_1.default.load(resourceId);
                    if (!resource)
                        throw new Error(`Cannot load note resource: ${n}`);
                }
                if (obj.tags) {
                    for (const tagTitle of obj.tags) {
                        const tag = await Tag_1.default.loadByTitle(tagTitle);
                        if (!tag)
                            throw new Error(`Cannot load note tag: ${tagTitle}`);
                        const hasNote = await Tag_1.default.hasNote(tag.id, note.id);
                        if (!hasNote)
                            throw new Error(`Tag not associated with note: ${tagTitle}`);
                    }
                }
            }
        }
    }
    await recurseCheck(data);
}
async function deploySyncTargetSnapshot(syncTargetType, syncVersion) {
    const sourceDir = `${snapshotBaseDir}/${syncVersion}/${syncTargetType}`;
    await fs.remove(test_utils_1.syncDir);
    await fs.copy(sourceDir, test_utils_1.syncDir);
}
async function main(syncTargetType) {
    shimInit({ sharp, nodeSqlite });
    const validSyncTargetTypes = ['normal', 'e2ee'];
    if (!validSyncTargetTypes.includes(syncTargetType))
        throw new Error(`Sync target type must be: ${validSyncTargetTypes.join(', ')}`);
    await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
    await (0, test_utils_1.switchClient)(1);
    await createTestData(exports.testData);
    if (syncTargetType === 'e2ee') {
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        await (0, test_utils_1.loadEncryptionMasterKey)();
    }
    await (0, test_utils_1.synchronizerStart)();
    await (0, test_utils_1.synchronizer)().start();
    if (!Setting_1.default.value('syncVersion'))
        throw new Error('syncVersion is not set');
    const destDir = `${snapshotBaseDir}/${Setting_1.default.value('syncVersion')}/${syncTargetType}`;
    await fs.mkdirp(destDir); // Create intermediate directories
    await fs.remove(destDir);
    await fs.mkdirp(destDir);
    await fs.copy(test_utils_1.syncDir, destDir);
    // eslint-disable-next-line no-console
    console.info(`Sync target snapshot created in: ${destDir}`);
}
//# sourceMappingURL=syncTargetUtils.js.map