"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TestApp = exports.logger = exports.syncDir = exports.simulateReadOnlyShareEnv = exports.createTestShareData = exports.createTempFile = exports.createNoteAndResource = exports.supportDir = exports.withWarningSilenced = exports.mockFetch = exports.runWithFakeTimers = exports.waitFor = exports.mockMobilePlatform = exports.newOcrService = exports.ocrSampleDir = void 0;
exports.createFolderTree = createFolderTree;
exports.naughtyStrings = naughtyStrings;
exports.waitForFolderCount = waitForFolderCount;
exports.afterAllCleanUp = afterAllCleanUp;
exports.exportDir = exportDir;
exports.synchronizerStart = synchronizerStart;
exports.afterEachCleanUp = afterEachCleanUp;
exports.syncTargetName = syncTargetName;
exports.setSyncTargetName = setSyncTargetName;
exports.createTempDir = createTempDir;
exports.isNetworkSyncTarget = isNetworkSyncTarget;
exports.kvStore = kvStore;
exports.expectThrow = expectThrow;
exports.expectNotThrow = expectNotThrow;
exports.resourceService = resourceService;
exports.resourceFetcher = resourceFetcher;
exports.tempFilePath = tempFilePath;
exports.allSyncTargetItemsEncrypted = allSyncTargetItemsEncrypted;
exports.msleep = msleep;
exports.setupDatabase = setupDatabase;
exports.revisionService = revisionService;
exports.setupDatabaseAndSynchronizer = setupDatabaseAndSynchronizer;
exports.db = db;
exports.synchronizer = synchronizer;
exports.fileApi = fileApi;
exports.sleep = sleep;
exports.clearDatabase = clearDatabase;
exports.switchClient = switchClient;
exports.syncTargetId = syncTargetId;
exports.objectsEqual = objectsEqual;
exports.checkThrowAsync = checkThrowAsync;
exports.checkThrow = checkThrow;
exports.encryptionService = encryptionService;
exports.loadEncryptionMasterKey = loadEncryptionMasterKey;
exports.fileContentEqual = fileContentEqual;
exports.decryptionWorker = decryptionWorker;
exports.currentClientId = currentClientId;
exports.id = id;
exports.ids = ids;
exports.sortedIds = sortedIds;
exports.at = at;
exports.createNTestNotes = createNTestNotes;
exports.createNTestFolders = createNTestFolders;
exports.createNTestTags = createNTestTags;
/* eslint-disable require-atomic-updates */
const BaseApplication_1 = require("../BaseApplication");
const BaseModel_1 = require("../BaseModel");
const Logger_1 = require("@joplin/utils/Logger");
const Setting_1 = require("../models/Setting");
const BaseService_1 = require("../services/BaseService");
const fs_driver_node_1 = require("../fs-driver-node");
const time_1 = require("../time");
const shim_1 = require("../shim");
const uuid_1 = require("../uuid");
const ResourceService_1 = require("../services/ResourceService");
const KeymapService_1 = require("../services/KeymapService");
const KvStore_1 = require("../services/KvStore");
const KeychainServiceDriver_node_1 = require("../services/keychain/KeychainServiceDriver.node");
const KeychainServiceDriver_dummy_1 = require("../services/keychain/KeychainServiceDriver.dummy");
const file_api_driver_joplinServer_1 = require("../file-api-driver-joplinServer");
const onedrive_api_1 = require("../onedrive-api");
const SyncTargetOneDrive_1 = require("../SyncTargetOneDrive");
const JoplinDatabase_1 = require("../JoplinDatabase");
const fs = require("fs-extra");
const { DatabaseDriverNode } = require('../database-driver-node.js');
const Folder_1 = require("../models/Folder");
const Note_1 = require("../models/Note");
const ItemChange_1 = require("../models/ItemChange");
const Resource_1 = require("../models/Resource");
const Tag_1 = require("../models/Tag");
const NoteTag_1 = require("../models/NoteTag");
const Revision_1 = require("../models/Revision");
const MasterKey_1 = require("../models/MasterKey");
const BaseItem_1 = require("../models/BaseItem");
const file_api_1 = require("../file-api");
const FileApiDriverMemory = require('../file-api-driver-memory').default;
const file_api_driver_local_1 = require("../file-api-driver-local");
const { FileApiDriverWebDav } = require('../file-api-driver-webdav.js');
const { FileApiDriverDropbox } = require('../file-api-driver-dropbox.js');
const { FileApiDriverOneDrive } = require('../file-api-driver-onedrive.js');
const SyncTargetRegistry_1 = require("../SyncTargetRegistry");
const SyncTargetMemory = require('../SyncTargetMemory.js');
const SyncTargetFilesystem_1 = require("../SyncTargetFilesystem");
const SyncTargetNextcloud = require('../SyncTargetNextcloud.js');
const SyncTargetDropbox = require('../SyncTargetDropbox.js');
const SyncTargetAmazonS3 = require('../SyncTargetAmazonS3.js');
const SyncTargetWebDAV = require('../SyncTargetWebDAV.js');
const SyncTargetJoplinServer_1 = require("../SyncTargetJoplinServer");
const EncryptionService_1 = require("../services/e2ee/EncryptionService");
const DecryptionWorker_1 = require("../services/DecryptionWorker");
const RevisionService_1 = require("../services/RevisionService");
const ResourceFetcher_1 = require("../services/ResourceFetcher");
const WebDavApi = require('../WebDavApi');
const DropboxApi = require('../DropboxApi');
const JoplinServerApi_1 = require("../JoplinServerApi");
const credentialFiles_1 = require("../utils/credentialFiles");
const SyncTargetJoplinCloud_1 = require("../SyncTargetJoplinCloud");
const KeychainService_1 = require("../services/keychain/KeychainService");
const SettingUtils_1 = require("../services/SettingUtils");
const syncInfoUtils_1 = require("../services/synchronizer/syncInfoUtils");
const SyncTargetNone_1 = require("../SyncTargetNone");
const ppk_1 = require("../services/e2ee/ppk/ppk");
const md5 = require('md5');
const { Dirnames } = require('../services/synchronizer/utils/types');
const RSA_node_1 = require("../services/e2ee/ppk/RSA.node");
const initLib_1 = require("../initLib");
const OcrDriverTesseract_1 = require("../services/ocr/drivers/OcrDriverTesseract");
const OcrService_1 = require("../services/ocr/OcrService");
const tesseract_js_1 = require("tesseract.js");
const registry_1 = require("../registry");
const path_1 = require("@joplin/utils/path");
const SyncTargetJoplinServerSAML_1 = require("../SyncTargetJoplinServerSAML");
const renderer_1 = require("@joplin/renderer");
const SearchEngine_1 = require("../services/search/SearchEngine");
// Each suite has its own separate data and temp directory so that multiple
// suites can be run at the same time. suiteName is what is used to
// differentiate between suite and it is currently set to a random string
// (Ideally it would be something like the filename currently being executed by
// Jest, to make debugging easier, but it's not clear how to get this info).
const suiteName_ = uuid_1.default.createNano();
const databases_ = [];
let synchronizers_ = [];
const fileApis_ = {};
const encryptionServices_ = [];
const revisionServices_ = [];
const decryptionWorkers_ = [];
const resourceServices_ = [];
const resourceFetchers_ = [];
const kvStores_ = [];
let currentClient_ = 1;
// The line `process.on('unhandledRejection'...` in all the test files is going to
// make it throw this error. It's not too big a problem so disable it for now.
// https://stackoverflow.com/questions/9768444/possible-eventemitter-memory-leak-detected
process.setMaxListeners(0);
shim_1.default.setIsTestingEnv(true);
const fsDriver = new fs_driver_node_1.default();
Logger_1.default.fsDriver_ = fsDriver;
Resource_1.default.fsDriver_ = fsDriver;
EncryptionService_1.default.fsDriver_ = fsDriver;
file_api_driver_local_1.default.fsDriver_ = fsDriver;
// Most test units were historically under /app-cli so most test-related
// directories are there but that should be moved eventually under the right
// packages, or even out of the monorepo for temp files, logs, etc.
const oldTestDir = `${__dirname}/../../app-cli/tests`;
const logDir = `${oldTestDir}/logs`;
const baseTempDir = `${oldTestDir}/tmp/${suiteName_}`;
const supportDir = `${oldTestDir}/support`;
exports.supportDir = supportDir;
exports.ocrSampleDir = `${oldTestDir}/ocr_samples`;
// We add a space in the data directory path as that will help uncover
// various space-in-path issues.
const dataDir = `${oldTestDir}/test data/${suiteName_}`;
const profileDir = `${dataDir}/profile`;
const rootProfileDir = profileDir;
fs.mkdirpSync(logDir);
fs.mkdirpSync(baseTempDir);
fs.mkdirpSync(dataDir);
fs.mkdirpSync(profileDir);
SyncTargetRegistry_1.default.addClass(SyncTargetNone_1.default);
SyncTargetRegistry_1.default.addClass(SyncTargetMemory);
SyncTargetRegistry_1.default.addClass(SyncTargetFilesystem_1.default);
SyncTargetRegistry_1.default.addClass(SyncTargetOneDrive_1.default);
SyncTargetRegistry_1.default.addClass(SyncTargetNextcloud);
SyncTargetRegistry_1.default.addClass(SyncTargetDropbox);
SyncTargetRegistry_1.default.addClass(SyncTargetAmazonS3);
SyncTargetRegistry_1.default.addClass(SyncTargetWebDAV);
SyncTargetRegistry_1.default.addClass(SyncTargetJoplinServer_1.default);
SyncTargetRegistry_1.default.addClass(SyncTargetJoplinServerSAML_1.default);
SyncTargetRegistry_1.default.addClass(SyncTargetJoplinCloud_1.default);
let syncTargetName_ = '';
let syncTargetId_ = null;
let sleepTime = 0;
let isNetworkSyncTarget_ = false;
function syncTargetName() {
    return syncTargetName_;
}
function setSyncTargetName(name) {
    if (name === syncTargetName_)
        return syncTargetName_;
    const previousName = syncTargetName_;
    syncTargetName_ = name;
    syncTargetId_ = SyncTargetRegistry_1.default.nameToId(syncTargetName_);
    sleepTime = syncTargetId_ === SyncTargetRegistry_1.default.nameToId('filesystem') ? 1001 : 100; // 400;
    isNetworkSyncTarget_ = ['nextcloud', 'dropbox', 'onedrive', 'amazon_s3', 'joplinServer', 'joplinServerSaml', 'joplinCloud'].includes(syncTargetName_);
    synchronizers_ = [];
    return previousName;
}
setSyncTargetName('memory');
// setSyncTargetName('filesystem');
// setSyncTargetName('nextcloud');
// setSyncTargetName('dropbox');
// setSyncTargetName('onedrive');
// setSyncTargetName('amazon_s3');
// setSyncTargetName('joplinServer');
// setSyncTargetName('joplinCloud');
// console.info(`Testing with sync target: ${syncTargetName_}`);
const syncDir = `${oldTestDir}/sync/${suiteName_}`;
exports.syncDir = syncDir;
// 90 seconds now that the tests are running in parallel and have been
// split into smaller suites might not be necessary but for now leave it
// anyway.
let defaultJestTimeout = 90 * 1000;
if (isNetworkSyncTarget_)
    defaultJestTimeout = 60 * 1000 * 10;
if (typeof jest !== 'undefined')
    jest.setTimeout(defaultJestTimeout);
const dbLogger = new Logger_1.default();
dbLogger.addTarget(Logger_1.TargetType.Console);
dbLogger.setLevel(Logger_1.default.LEVEL_WARN);
const logger = new Logger_1.default();
exports.logger = logger;
logger.addTarget(Logger_1.TargetType.Console);
logger.setLevel(Logger_1.LogLevel.Warn); // Set to DEBUG to display sync process in console
Logger_1.default.initializeGlobalLogger(logger);
(0, initLib_1.default)(logger);
BaseItem_1.default.loadClass('Note', Note_1.default);
BaseItem_1.default.loadClass('Folder', Folder_1.default);
BaseItem_1.default.loadClass('Resource', Resource_1.default);
BaseItem_1.default.loadClass('Tag', Tag_1.default);
BaseItem_1.default.loadClass('NoteTag', NoteTag_1.default);
BaseItem_1.default.loadClass('MasterKey', MasterKey_1.default);
BaseItem_1.default.loadClass('Revision', Revision_1.default);
Setting_1.default.setConstant('appId', 'net.cozic.joplintest-cli');
Setting_1.default.setConstant('appType', Setting_1.AppType.Cli);
Setting_1.default.setConstant('tempDir', baseTempDir);
Setting_1.default.setConstant('cacheDir', baseTempDir);
Setting_1.default.setConstant('resourceDir', baseTempDir);
Setting_1.default.setConstant('pluginDataDir', `${profileDir}/profile/plugin-data`);
Setting_1.default.setConstant('pluginAssetDir', `${(0, path_1.dirname)(require.resolve('@joplin/renderer'))}/assets`);
Setting_1.default.setConstant('profileDir', profileDir);
Setting_1.default.setConstant('rootProfileDir', rootProfileDir);
Setting_1.default.setConstant('env', Setting_1.Env.Dev);
BaseService_1.default.logger_ = logger;
Setting_1.default.autoSaveEnabled = false;
function syncTargetId() {
    return syncTargetId_;
}
function isNetworkSyncTarget() {
    return isNetworkSyncTarget_;
}
function sleep(n) {
    return new Promise((resolve) => {
        shim_1.default.setTimeout(() => {
            resolve(null);
        }, Math.round(n * 1000));
    });
}
function msleep(ms) {
    // It seems setTimeout can sometimes last less time than the provided
    // interval:
    //
    // https://stackoverflow.com/a/50912029/561309
    //
    // This can cause issues in tests where we expect the actual duration to be
    // the same as the provided interval or more, but not less. So the code
    // below check that the elapsed time is no less than the provided interval,
    // and if it is, it waits a bit longer.
    const startTime = Date.now();
    return new Promise((resolve) => {
        shim_1.default.setTimeout(() => {
            if (Date.now() - startTime < ms) {
                const iid = setInterval(() => {
                    if (Date.now() - startTime >= ms) {
                        clearInterval(iid);
                        resolve(null);
                    }
                }, 2);
            }
            else {
                resolve(null);
            }
        }, ms);
    });
}
function currentClientId() {
    return currentClient_;
}
async function afterEachCleanUp() {
    await ItemChange_1.default.waitForAllSaved();
    KeymapService_1.default.destroyInstance();
}
async function afterAllCleanUp() {
    if (fileApi()) {
        try {
            await fileApi().clearRoot();
        }
        catch (error) {
            console.warn('Could not clear sync target root:', error);
        }
    }
}
const settingFilename = (id) => {
    return `settings-${id}.json`;
};
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
async function switchClient(id, options = null) {
    options = Object.assign({ keychainEnabled: false }, options);
    if (!databases_[id])
        throw new Error(`Call setupDatabaseAndSynchronizer(${id}) first!!`);
    await time_1.default.msleep(sleepTime); // Always leave a little time so that updated_time properties don't overlap
    await Setting_1.default.saveAll();
    currentClient_ = id;
    BaseModel_1.default.setDb(databases_[id]);
    KvStore_1.default.instance().setDb(databases_[id]);
    SearchEngine_1.default.instance().setDb(databases_[id]);
    BaseItem_1.default.encryptionService_ = encryptionServices_[id];
    Resource_1.default.encryptionService_ = encryptionServices_[id];
    BaseItem_1.default.revisionService_ = revisionServices_[id];
    ResourceFetcher_1.default.instance_ = resourceFetchers_[id];
    DecryptionWorker_1.default.instance_ = decryptionWorker(id);
    await Setting_1.default.reset();
    Setting_1.default.settingFilename = settingFilename(id);
    Setting_1.default.setConstant('profileDir', rootProfileDir);
    Setting_1.default.setConstant('rootProfileDir', rootProfileDir);
    Setting_1.default.setConstant('resourceDirName', resourceDirName(id));
    Setting_1.default.setConstant('resourceDir', resourceDir(id));
    Setting_1.default.setConstant('pluginDir', pluginDir(id));
    Setting_1.default.setConstant('isSubProfile', false);
    await (0, SettingUtils_1.loadKeychainServiceAndSettings)(options.keychainEnabled ? [KeychainServiceDriver_node_1.default] : []);
    Setting_1.default.setValue('sync.target', syncTargetId());
    Setting_1.default.setValue('sync.wipeOutFailSafe', false); // To keep things simple, always disable fail-safe unless explicitly set in the test itself
    // More generally, this function should clear all data, and so that should
    // include settings.json
    await clearSettingFile(id);
}
async function clearDatabase(id = null) {
    if (id === null)
        id = currentClient_;
    if (!databases_[id])
        return;
    await ItemChange_1.default.waitForAllSaved();
    const tableNames = [
        'deleted_items',
        'folders',
        'item_changes',
        'items_normalized',
        'key_values',
        'master_keys',
        'note_resources',
        'note_tags',
        'notes_normalized',
        'notes',
        'resources',
        'revisions',
        'settings',
        'sync_items',
        'tags',
    ];
    const queries = [];
    for (const n of tableNames) {
        queries.push(`DELETE FROM ${n}`);
        queries.push(`DELETE FROM sqlite_sequence WHERE name="${n}"`); // Reset autoincremented IDs
    }
    await databases_[id].transactionExecBatch(queries);
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
async function setupDatabase(id = null, options = null) {
    options = Object.assign({ keychainEnabled: false }, options);
    if (id === null)
        id = currentClient_;
    Setting_1.default.cancelScheduleSave();
    // Note that this was changed from `Setting.cache_ = []` to `await
    // Setting.reset()` during the TypeScript conversion. Normally this is
    // more correct but something to keep in mind anyway in case there are
    // some strange async issue related to settings when the tests are
    // running.
    await Setting_1.default.reset();
    Setting_1.default.setConstant('profileDir', rootProfileDir);
    Setting_1.default.setConstant('rootProfileDir', rootProfileDir);
    Setting_1.default.setConstant('isSubProfile', false);
    if (databases_[id]) {
        BaseModel_1.default.setDb(databases_[id]);
        await clearDatabase(id);
        await (0, SettingUtils_1.loadKeychainServiceAndSettings)(options.keychainEnabled ? [KeychainServiceDriver_node_1.default] : []);
        Setting_1.default.setValue('sync.target', syncTargetId());
        return;
    }
    const filePath = `${dataDir}/test-${id}.sqlite`;
    try {
        await fs.unlink(filePath);
    }
    catch (error) {
        // Don't care if the file doesn't exist
    }
    databases_[id] = new JoplinDatabase_1.default(new DatabaseDriverNode());
    databases_[id].setLogger(dbLogger);
    await databases_[id].open({ name: filePath });
    BaseModel_1.default.setDb(databases_[id]);
    await clearSettingFile(id);
    await (0, SettingUtils_1.loadKeychainServiceAndSettings)([options.keychainEnabled ? KeychainServiceDriver_node_1.default : KeychainServiceDriver_dummy_1.default]);
    registry_1.reg.setDb(databases_[id]);
    Setting_1.default.setValue('sync.target', syncTargetId());
}
async function clearSettingFile(id) {
    Setting_1.default.settingFilename = `settings-${id}.json`;
    await fs.remove(Setting_1.default.settingFilePath);
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
async function createFolderTree(parentId, tree, num = 0) {
    let rootFolder = null;
    for (const item of tree) {
        const isFolder = !!item.children;
        num++;
        const data = Object.assign({}, item);
        delete data.children;
        if (isFolder) {
            const folder = await Folder_1.default.save(Object.assign({ title: `Folder ${num}`, parent_id: parentId }, data));
            if (!rootFolder)
                rootFolder = folder;
            if (item.children.length)
                await createFolderTree(folder.id, item.children, num);
        }
        else {
            await Note_1.default.save(Object.assign({ title: `Note ${num}`, parent_id: parentId }, data));
        }
    }
    return rootFolder;
}
function exportDir(id = null) {
    if (id === null)
        id = currentClient_;
    return `${dataDir}/export`;
}
function resourceDirName(id = null) {
    if (id === null)
        id = currentClient_;
    return `resources-${id}`;
}
function resourceDir(id = null) {
    if (id === null)
        id = currentClient_;
    return `${dataDir}/${resourceDirName(id)}`;
}
function pluginDir(id = null) {
    if (id === null)
        id = currentClient_;
    return `${dataDir}/plugins-${id}`;
}
const createNoteAndResource = async (options = null) => {
    options = Object.assign({ path: `${supportDir}/photo.jpg`, markupLanguage: renderer_1.MarkupLanguage.Markdown }, options);
    let note = await Note_1.default.save({ markup_language: options.markupLanguage });
    note = await shim_1.default.attachFileToNote(note, options.path);
    const resourceIds = await Note_1.default.linkedItemIds(note.body);
    const resource = await Resource_1.default.load(resourceIds[0]);
    return { note, resource };
};
exports.createNoteAndResource = createNoteAndResource;
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
async function setupDatabaseAndSynchronizer(id, options = null) {
    if (id === null)
        id = currentClient_;
    BaseService_1.default.logger_ = logger;
    await setupDatabase(id, options);
    EncryptionService_1.default.instance_ = null;
    DecryptionWorker_1.default.instance_ = null;
    await fs.remove(resourceDir(id));
    await fs.mkdirp(resourceDir(id));
    await fs.remove(pluginDir(id));
    await fs.mkdirp(pluginDir(id));
    if (!synchronizers_[id]) {
        const SyncTargetClass = SyncTargetRegistry_1.default.classById(syncTargetId_);
        const syncTarget = new SyncTargetClass(db(id));
        await initFileApi();
        syncTarget.setFileApi(fileApi());
        syncTarget.setLogger(logger);
        synchronizers_[id] = await syncTarget.synchronizer();
        // For now unset the share service as it's not properly initialised.
        // Share service tests are in ShareService.test.ts normally, and if it
        // becomes necessary to test integration with the synchroniser we can
        // initialize it here.
        synchronizers_[id].setShareService(null);
    }
    encryptionServices_[id] = new EncryptionService_1.default();
    revisionServices_[id] = new RevisionService_1.default();
    decryptionWorkers_[id] = new DecryptionWorker_1.default();
    decryptionWorkers_[id].setEncryptionService(encryptionServices_[id]);
    resourceServices_[id] = new ResourceService_1.default();
    resourceFetchers_[id] = new ResourceFetcher_1.default(() => { return synchronizers_[id].api(); });
    kvStores_[id] = new KvStore_1.default();
    (0, ppk_1.setRSA)(RSA_node_1.default);
    await fileApi().initialize();
    await fileApi().clearRoot();
}
function db(id = null) {
    if (id === null)
        id = currentClient_;
    return databases_[id];
}
function synchronizer(id = null) {
    if (id === null)
        id = currentClient_;
    return synchronizers_[id];
}
// This is like calling synchronizer.start() but it handles the
// complexity of passing around the sync context depending on
// the client.
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
async function synchronizerStart(id = null, extraOptions = null) {
    if (id === null)
        id = currentClient_;
    const contextKey = `sync.${syncTargetId()}.context`;
    const contextString = Setting_1.default.value(contextKey);
    const context = contextString ? JSON.parse(contextString) : {};
    const options = Object.assign({}, extraOptions);
    if (context)
        options.context = context;
    const newContext = await synchronizer(id).start(options);
    Setting_1.default.setValue(contextKey, JSON.stringify(newContext));
    return newContext;
}
function encryptionService(id = null) {
    if (id === null)
        id = currentClient_;
    return encryptionServices_[id];
}
function kvStore(id = null) {
    if (id === null)
        id = currentClient_;
    const o = kvStores_[id];
    o.setDb(db(id));
    return o;
}
function revisionService(id = null) {
    if (id === null)
        id = currentClient_;
    return revisionServices_[id];
}
function decryptionWorker(id = null) {
    if (id === null)
        id = currentClient_;
    const o = decryptionWorkers_[id];
    o === null || o === void 0 ? void 0 : o.setKvStore(kvStore(id));
    return o;
}
function resourceService(id = null) {
    if (id === null)
        id = currentClient_;
    return resourceServices_[id];
}
function resourceFetcher(id = null) {
    if (id === null)
        id = currentClient_;
    return resourceFetchers_[id];
}
async function loadEncryptionMasterKey(id = null, useExisting = false) {
    const service = encryptionService(id);
    const password = '123456';
    let masterKey = null;
    if (!useExisting) { // Create it
        masterKey = await service.generateMasterKey(password);
        masterKey = await MasterKey_1.default.save(masterKey);
    }
    else { // Use the one already available
        const masterKeys = await MasterKey_1.default.all();
        if (!masterKeys.length)
            throw new Error('No master key available');
        masterKey = masterKeys[0];
    }
    const passwordCache = Setting_1.default.value('encryption.passwordCache');
    passwordCache[masterKey.id] = password;
    Setting_1.default.setValue('encryption.passwordCache', passwordCache);
    await Setting_1.default.saveAll();
    await service.loadMasterKey(masterKey, password, true);
    (0, syncInfoUtils_1.setActiveMasterKeyId)(masterKey.id);
    return masterKey;
}
function mustRunInBand() {
    if (!process.argv.includes('--runInBand')) {
        throw new Error('Tests must be run sequentially for this sync target, with the --runInBand arg. eg `yarn test --runInBand`');
    }
}
async function initFileApi() {
    if (fileApis_[syncTargetId_])
        return;
    let fileApi = null;
    if (syncTargetId_ === SyncTargetRegistry_1.default.nameToId('filesystem')) {
        fs.removeSync(syncDir);
        fs.mkdirpSync(syncDir);
        fileApi = new file_api_1.FileApi(syncDir, new file_api_driver_local_1.default());
    }
    else if (syncTargetId_ === SyncTargetRegistry_1.default.nameToId('memory')) {
        fileApi = new file_api_1.FileApi('/root', new FileApiDriverMemory());
    }
    else if (syncTargetId_ === SyncTargetRegistry_1.default.nameToId('nextcloud')) {
        const options = require(`${oldTestDir}/support/nextcloud-auth.json`);
        const api = new WebDavApi({
            baseUrl: () => options.baseUrl,
            username: () => options.username,
            password: () => options.password,
        });
        fileApi = new file_api_1.FileApi('', new FileApiDriverWebDav(api));
    }
    else if (syncTargetId_ === SyncTargetRegistry_1.default.nameToId('dropbox')) {
        // To get a token, go to the App Console:
        // https://www.dropbox.com/developers/apps/
        // Then select "JoplinTest" and click "Generated access token"
        const api = new DropboxApi();
        const authTokenPath = `${oldTestDir}/support/dropbox-auth.txt`;
        const authToken = fs.readFileSync(authTokenPath, 'utf8');
        if (!authToken)
            throw new Error(`Dropbox auth token missing in ${authTokenPath}`);
        api.setAuthToken(authToken);
        fileApi = new file_api_1.FileApi('', new FileApiDriverDropbox(api));
    }
    else if (syncTargetId_ === SyncTargetRegistry_1.default.nameToId('onedrive')) {
        // To get a token, open the URL below corresponding to your account type,
        // then copy the *complete* redirection URL in onedrive-auth.txt. Keep in mind that auth
        // data only lasts 1h for OneDrive.
        //
        // Personal OneDrive Account:
        // https://login.live.com/oauth20_authorize.srf?client_id=f1e68e1e-a729-4514-b041-4fdd5c7ac03a&scope=files.readwrite,offline_access&response_type=token&redirect_uri=https://joplinapp.org
        //
        // Business OneDrive Account:
        // https://login.microsoftonline.com/common/oauth2/v2.0/authorize?client_id=f1e68e1e-a729-4514-b041-4fdd5c7ac03a&scope=files.readwrite offline_access&response_type=token&redirect_uri=https://joplinapp.org
        //
        // Also for now OneDrive tests cannot be run in parallel because
        // for that each suite would need its own sub-directory within the
        // OneDrive app directory, and it's not clear how to get that
        // working.
        mustRunInBand();
        const { parameters, setEnvOverride } = require('../parameters.js');
        Setting_1.default.setConstant('env', Setting_1.Env.Dev);
        setEnvOverride('test');
        const config = parameters().oneDriveTest;
        const api = new onedrive_api_1.default(config.id, config.secret, false);
        const authData = fs.readFileSync(await (0, credentialFiles_1.credentialFile)('onedrive-auth.txt'), 'utf8');
        const urlInfo = require('url-parse')(authData, true);
        const auth = require('querystring').parse(urlInfo.hash.substr(1));
        api.setAuth(auth);
        const accountProperties = await api.execAccountPropertiesRequest();
        api.setAccountProperties(accountProperties);
        const appDir = await api.appDirectory();
        fileApi = new file_api_1.FileApi(appDir, new FileApiDriverOneDrive(api));
    }
    else if (syncTargetId_ === SyncTargetRegistry_1.default.nameToId('amazon_s3')) {
        // (Most of?) the @aws-sdk libraries depend on an old version of uuid
        // that doesn't work with jest (without converting ES6 exports to CommonJS).
        //
        // Require it dynamically so that this doesn't break test environments that
        // aren't configured to do this conversion.
        const { FileApiDriverAmazonS3 } = require('../file-api-driver-amazon-s3.js');
        const { S3Client } = require('@aws-sdk/client-s3');
        // We make sure for S3 tests run in band because tests
        // share the same directory which will cause locking errors.
        mustRunInBand();
        const amazonS3CredsPath = `${oldTestDir}/support/amazon-s3-auth.json`;
        const amazonS3Creds = require(amazonS3CredsPath);
        if (!amazonS3Creds || !amazonS3Creds.credentials)
            throw new Error(`AWS auth JSON missing in ${amazonS3CredsPath} format should be: { "credentials": { "accessKeyId": "", "secretAccessKey": "", } "bucket": "mybucket", region: "", forcePathStyle: ""}`);
        const api = new S3Client({ region: amazonS3Creds.region, credentials: amazonS3Creds.credentials, s3UseArnRegion: true, forcePathStyle: amazonS3Creds.forcePathStyle, endpoint: amazonS3Creds.endpoint });
        fileApi = new file_api_1.FileApi('', new FileApiDriverAmazonS3(api, amazonS3Creds.bucket));
    }
    else if (syncTargetId_ === SyncTargetRegistry_1.default.nameToId('joplinServer') || syncTargetId_ === SyncTargetRegistry_1.default.nameToId('joplinCloud')) {
        mustRunInBand();
        const joplinServerAuth = JSON.parse(await (0, credentialFiles_1.readCredentialFile)('joplin-server-test-units-2.json'));
        // const joplinServerAuth = {
        //     "email": "admin@localhost",
        //     "password": "admin",
        //     "baseUrl": "http://api.joplincloud.local:22300",
        //     "userContentBaseUrl": ""
        // }
        // Note that to test the API in parallel mode, you need to use Postgres
        // as database, as the SQLite database is not reliable when being
        // read/write from multiple processes at the same time.
        const api = new JoplinServerApi_1.default({
            baseUrl: () => joplinServerAuth.baseUrl,
            userContentBaseUrl: () => joplinServerAuth.userContentBaseUrl,
            username: () => joplinServerAuth.email,
            password: () => joplinServerAuth.password,
            apiKey: () => '',
            session: () => null,
        });
        fileApi = new file_api_1.FileApi('', new file_api_driver_joplinServer_1.default(api));
    }
    fileApi.setLogger(logger);
    fileApi.setSyncTargetId(syncTargetId_);
    fileApi.setTempDirName(Dirnames.Temp);
    fileApi.requestRepeatCount_ = isNetworkSyncTarget_ ? 1 : 0;
    fileApis_[syncTargetId_] = fileApi;
}
function fileApi() {
    return fileApis_[syncTargetId_];
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
function objectsEqual(o1, o2) {
    if (Object.getOwnPropertyNames(o1).length !== Object.getOwnPropertyNames(o2).length)
        return false;
    for (const n in o1) {
        if (!o1.hasOwnProperty(n))
            continue;
        if (o1[n] !== o2[n])
            return false;
    }
    return true;
}
// eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
async function checkThrowAsync(asyncFn) {
    let hasThrown = false;
    try {
        await asyncFn();
    }
    catch (error) {
        hasThrown = true;
    }
    return hasThrown;
}
// eslint-disable-next-line @typescript-eslint/ban-types, @typescript-eslint/no-explicit-any -- Old code before rule was applied, Old code before rule was applied
async function expectThrow(asyncFn, errorCode = undefined, errorMessage = undefined) {
    let hasThrown = false;
    let thrownError = null;
    try {
        await asyncFn();
    }
    catch (error) {
        hasThrown = true;
        thrownError = error;
    }
    if (!hasThrown) {
        expect('not throw').toBe('throw');
    }
    else if (errorMessage !== undefined) {
        if (thrownError.message !== errorMessage) {
            expect(`error message: ${thrownError.message}`).toBe(`error message: ${errorMessage}`);
        }
        else {
            expect(true).toBe(true);
        }
    }
    else if (thrownError.code !== errorCode) {
        console.error(thrownError);
        expect(`error code: ${thrownError.code}`).toBe(`error code: ${errorCode}`);
    }
    else {
        expect(true).toBe(true);
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
async function expectNotThrow(asyncFn) {
    let thrownError = null;
    try {
        await asyncFn();
    }
    catch (error) {
        thrownError = error;
    }
    if (thrownError) {
        console.error(thrownError);
        expect(thrownError.message).toBe('');
    }
    else {
        expect(true).toBe(true);
    }
}
// eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
function checkThrow(fn) {
    let hasThrown = false;
    try {
        fn();
    }
    catch (error) {
        hasThrown = true;
    }
    return hasThrown;
}
function fileContentEqual(path1, path2) {
    const fs = require('fs-extra');
    const content1 = fs.readFileSync(path1, 'base64');
    const content2 = fs.readFileSync(path2, 'base64');
    return content1 === content2;
}
async function allSyncTargetItemsEncrypted() {
    const list = await fileApi().list('', { includeDirs: false });
    const files = list.items;
    let totalCount = 0;
    let encryptedCount = 0;
    for (let i = 0; i < files.length; i++) {
        const file = files[i];
        if (!BaseItem_1.default.isSystemPath(file.path))
            continue;
        const remoteContentString = await fileApi().get(file.path);
        const remoteContent = await BaseItem_1.default.unserialize(remoteContentString);
        const ItemClass = BaseItem_1.default.itemClass(remoteContent);
        if (!ItemClass.encryptionSupported())
            continue;
        totalCount++;
        if (remoteContent.type_ === BaseModel_1.default.TYPE_RESOURCE) {
            const content = await fileApi().get(`.resource/${remoteContent.id}`);
            totalCount++;
            if (content.substr(0, 5) === 'JED01')
                encryptedCount++;
        }
        if (remoteContent.encryption_applied)
            encryptedCount++;
    }
    if (!totalCount)
        throw new Error('No encryptable item on sync target');
    return totalCount === encryptedCount;
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
function id(a) {
    return a.id;
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
function ids(a) {
    return a.map(n => n.id);
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
function sortedIds(a) {
    return ids(a).sort();
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
function at(a, indexes) {
    const out = [];
    for (let i = 0; i < indexes.length; i++) {
        out.push(a[indexes[i]]);
    }
    return out;
}
async function createNTestFolders(n) {
    const folders = [];
    for (let i = 0; i < n; i++) {
        const folder = await Folder_1.default.save({ title: 'folder' });
        folders.push(folder);
        await time_1.default.msleep(10);
    }
    return folders;
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
async function createNTestNotes(n, folder, tagIds = null, title = 'note') {
    const notes = [];
    for (let i = 0; i < n; i++) {
        const title_ = n > 1 ? `${title}${i}` : title;
        const note = await Note_1.default.save({ title: title_, parent_id: folder.id, is_conflict: 0, deleted_time: 0 });
        notes.push(note);
        await time_1.default.msleep(10);
    }
    if (tagIds) {
        for (let i = 0; i < notes.length; i++) {
            await Tag_1.default.setNoteTagsByIds(notes[i].id, tagIds);
            await time_1.default.msleep(10);
        }
    }
    return notes;
}
async function createNTestTags(n) {
    const tags = [];
    for (let i = 0; i < n; i++) {
        const tag = await Tag_1.default.save({ title: 'tag' });
        tags.push(tag);
        await time_1.default.msleep(10);
    }
    return tags;
}
function tempFilePath(ext) {
    return `${Setting_1.default.value('tempDir')}/${md5(Date.now() + Math.random())}.${ext}`;
}
const createTempFile = async (content = '') => {
    const path = tempFilePath('txt');
    await fs.writeFile(path, content, 'utf8');
    return path;
};
exports.createTempFile = createTempFile;
async function createTempDir() {
    const tempDirPath = `${baseTempDir}/${uuid_1.default.createNano()}`;
    await fs.mkdirp(tempDirPath);
    return tempDirPath;
}
async function waitForFolderCount(count) {
    const timeout = 2000;
    const startTime = Date.now();
    while (true) {
        const folders = await Folder_1.default.all();
        if (folders.length >= count)
            return;
        if (Date.now() - startTime > timeout)
            throw new Error('Timeout waiting for folders to be created');
        await msleep(10);
    }
}
let naughtyStrings_ = null;
async function naughtyStrings() {
    if (naughtyStrings_)
        return naughtyStrings_;
    const t = await fs.readFile(`${supportDir}/big-list-of-naughty-strings.txt`, 'utf8');
    const lines = t.split('\n');
    naughtyStrings_ = [];
    for (const line of lines) {
        const trimmed = line.trim();
        if (!trimmed)
            continue;
        if (trimmed.indexOf('#') === 0)
            continue;
        naughtyStrings_.push(line);
    }
    return naughtyStrings_;
}
// TODO: Update for Jest
// function mockDate(year, month, day, tick) {
// 	const fixedDate = new Date(2020, 0, 1);
// 	jasmine.clock().install();
// 	jasmine.clock().mockDate(fixedDate);
// }
// function restoreDate() {
// 	jasmine.clock().uninstall();
// }
// Application for feature integration testing
class TestApp extends BaseApplication_1.default {
    constructor(hasGui = true) {
        KeychainService_1.default.instance().enabled = false;
        super();
        this.hasGui_ = hasGui;
        this.middlewareCalls_ = [];
        this.logger_ = super.logger();
    }
    hasGui() {
        return this.hasGui_;
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async start(argv) {
        this.logger_.info('Test app starting...');
        if (!argv.includes('--profile')) {
            argv = argv.concat(['--profile', `tests-build/profile/${uuid_1.default.create()}`]);
        }
        argv = await super.start(['', ''].concat(argv), { setupGlobalLogger: false });
        // For now, disable sync and encryption to avoid spurious intermittent failures
        // caused by them interupting processing and causing delays.
        Setting_1.default.setValue('sync.interval', 0);
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        this.initRedux();
        Setting_1.default.dispatchUpdateAll();
        await ItemChange_1.default.waitForAllSaved();
        await this.wait();
        this.logger_.info('Test app started...');
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async generalMiddleware(store, next, action) {
        this.middlewareCalls_.push(true);
        try {
            await super.generalMiddleware(store, next, action);
        }
        finally {
            this.middlewareCalls_.pop();
        }
    }
    async wait() {
        return new Promise((resolve) => {
            const iid = shim_1.default.setInterval(() => {
                if (!this.middlewareCalls_.length) {
                    clearInterval(iid);
                    resolve(null);
                }
            }, 100);
        });
    }
    async profileDir() {
        return Setting_1.default.value('profileDir');
    }
    async destroy() {
        this.logger_.info('Test app stopping...');
        await this.wait();
        await ItemChange_1.default.waitForAllSaved();
        this.deinitRedux();
        await super.destroy();
        await time_1.default.msleep(100);
    }
}
exports.TestApp = TestApp;
const createTestShareData = (shareId) => {
    const share = {
        id: shareId,
        folder_id: '',
        master_key_id: '',
        note_id: '',
        type: 1,
    };
    return {
        processingShareInvitationResponse: false,
        shares: [share],
        shareInvitations: [
            {
                id: '',
                master_key: {},
                status: 0,
                share,
                can_read: 1,
                can_write: 0,
            },
        ],
        shareUsers: {},
    };
};
exports.createTestShareData = createTestShareData;
const mergeShareData = (state1, state2) => {
    return Object.assign(Object.assign({}, state1), { shares: [...state1.shares, ...state2.shares], shareInvitations: [
            ...state1.shareInvitations,
            ...state2.shareInvitations,
        ], shareUsers: Object.assign(Object.assign({}, state1.shareUsers), state2.shareUsers) });
};
const simulateReadOnlyShareEnv = (shareIds, store) => {
    if (!Array.isArray(shareIds)) {
        shareIds = [shareIds];
    }
    Setting_1.default.setValue('sync.target', 10);
    Setting_1.default.setValue('sync.userId', 'abcd');
    // Create all shares
    let shareData = null;
    for (const shareId of shareIds) {
        const newShareData = createTestShareData(shareId);
        if (!shareData) {
            shareData = newShareData;
        }
        else {
            shareData = mergeShareData(shareData, newShareData);
        }
    }
    BaseItem_1.default.syncShareCache = shareData;
    if (store) {
        store.dispatch({
            type: 'SHARE_SET',
            shares: shareData.shares,
        });
        store.dispatch({
            type: 'SHARE_INVITATION_SET',
            shareInvitations: shareData.shareInvitations,
        });
        store.dispatch({
            type: 'SHARE_USER_SET',
            shareUsers: shareData.shareUsers,
        });
    }
    return () => {
        BaseItem_1.default.syncShareCache = null;
        Setting_1.default.setValue('sync.userId', '');
    };
};
exports.simulateReadOnlyShareEnv = simulateReadOnlyShareEnv;
const newOcrService = () => {
    const driver = new OcrDriverTesseract_1.default({ createWorker: tesseract_js_1.createWorker }, { workerPath: null, corePath: null, languageDataPath: null });
    return new OcrService_1.default([driver]);
};
exports.newOcrService = newOcrService;
const mockMobilePlatform = (platform) => {
    const originalMobilePlatform = shim_1.default.mobilePlatform;
    const originalIsNode = shim_1.default.isNode;
    shim_1.default.mobilePlatform = () => platform;
    shim_1.default.isNode = () => false;
    return {
        reset: () => {
            shim_1.default.mobilePlatform = originalMobilePlatform;
            shim_1.default.isNode = originalIsNode;
        },
    };
};
exports.mockMobilePlatform = mockMobilePlatform;
// Waits for callback to not throw. Similar to react-native-testing-library's waitFor, but works better
// with Joplin's mix of real and fake Jest timers.
const realSetTimeout = setTimeout;
const waitFor = async (callback) => {
    const timeout = 10000;
    const startTime = performance.now();
    let passed = false;
    let lastError = null;
    while (!passed && performance.now() - startTime < timeout) {
        try {
            await callback();
            passed = true;
            lastError = null;
        }
        catch (error) {
            lastError = error;
            await new Promise(resolve => {
                realSetTimeout(() => resolve(), 10);
            });
        }
    }
    if (lastError) {
        throw lastError;
    }
};
exports.waitFor = waitFor;
const runWithFakeTimers = async (callback) => {
    if (typeof jest === 'undefined') {
        throw new Error('Fake timers are only supported in jest.');
    }
    // advanceTimers: Needed by Joplin's database driver
    jest.useFakeTimers({ advanceTimers: true });
    // The shim.setTimeout and similar functions need to be changed to
    // use fake timers.
    const originalSetTimeout = shim_1.default.setTimeout;
    const originalSetInterval = shim_1.default.setInterval;
    const originalClearTimeout = shim_1.default.clearTimeout;
    const originalClearInterval = shim_1.default.clearInterval;
    shim_1.default.setTimeout = setTimeout;
    shim_1.default.setInterval = setInterval;
    shim_1.default.clearInterval = clearInterval;
    shim_1.default.clearTimeout = clearTimeout;
    try {
        return await callback();
    }
    finally {
        jest.runOnlyPendingTimers();
        shim_1.default.setTimeout = originalSetTimeout;
        shim_1.default.setInterval = originalSetInterval;
        shim_1.default.clearTimeout = originalClearTimeout;
        shim_1.default.clearInterval = originalClearInterval;
        jest.useRealTimers();
    }
};
exports.runWithFakeTimers = runWithFakeTimers;
// Mocks shim.fetch, but may not mock other fetch-related methods
const mockFetch = (requestHandler) => {
    const originalFetch = shim_1.default.fetch;
    shim_1.default.fetch = (url, options) => {
        const request = new Request(url, options);
        const mockResponse = requestHandler(request);
        if (mockResponse) {
            return Promise.resolve(mockResponse);
        }
        else {
            return originalFetch(url, options);
        }
    };
    return {
        reset: () => {
            shim_1.default.fetch = originalFetch;
        },
    };
};
exports.mockFetch = mockFetch;
const withWarningSilenced = async (warningRegex, task) => {
    // See https://jestjs.io/docs/jest-object#spied-methods-and-the-using-keyword, which
    // shows how to use .spyOn to hide warnings
    let warningMock;
    try {
        warningMock = jest.spyOn(console, 'warn');
        warningMock.mockImplementation((message, ...args) => {
            const fullMessage = [message, ...args].join(' ');
            if (!fullMessage.match(warningRegex)) {
                console.error(`Unexpected warning: ${message}`, ...args);
            }
        });
        return await task();
    }
    finally {
        warningMock.mockRestore();
    }
};
exports.withWarningSilenced = withWarningSilenced;
//# sourceMappingURL=test-utils.js.map