"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.objectToQueryString = exports.extractResourceUrls = exports.fileUrlToResourceUrl = exports.parseResourceUrl = exports.isResourceUrl = exports.prependBaseUrl = exports.urlProtocol = exports.urlWithoutPath = exports.hash = void 0;
const path_utils_1 = require("./path-utils");
const string_utils_1 = require("./string-utils");
const hash = (url) => {
    const s = url.split('#');
    if (s.length <= 1)
        return '';
    return s[s.length - 1];
};
exports.hash = hash;
const urlWithoutPath = (url) => {
    const parsed = new URL(url);
    return `${parsed.protocol}//${parsed.host}`;
};
exports.urlWithoutPath = urlWithoutPath;
const urlProtocol = (url) => {
    if (!url)
        return '';
    let parsed;
    try {
        parsed = new URL(url);
    }
    catch (error) {
        // Match the NodeJS url.parse behavior in the case of an invalid URL:
        return null;
    }
    return parsed.protocol;
};
exports.urlProtocol = urlProtocol;
const prependBaseUrl = (url, baseUrl) => {
    baseUrl = (0, path_utils_1.rtrimSlashes)(baseUrl).trim(); // All the code below assumes that the baseUrl does not end up with a slash
    url = url.trim();
    if (!url)
        url = '';
    if (!baseUrl)
        return url;
    if (url.indexOf('#') === 0)
        return url; // Don't prepend if it's a local anchor
    if ((0, exports.urlProtocol)(url))
        return url; // Don't prepend the base URL if the URL already has a scheme
    if (url.length >= 2 && url.indexOf('//') === 0) {
        // If it starts with // it's a protcol-relative URL
        return (0, exports.urlProtocol)(baseUrl) + url;
    }
    else if (url && url[0] === '/') {
        // If it starts with a slash, it's an absolute URL so it should be relative to the domain (and not to the full baseUrl)
        return (0, exports.urlWithoutPath)(baseUrl) + url;
    }
    else {
        return baseUrl + (url ? `/${url}` : '');
    }
};
exports.prependBaseUrl = prependBaseUrl;
const resourceRegex = /^(joplin:\/\/|:\/)([0-9a-zA-Z]{32})(|#[^\s]*)(|\s".*?")$/;
const isResourceUrl = (url) => {
    return !!url.match(resourceRegex);
};
exports.isResourceUrl = isResourceUrl;
const parseResourceUrl = (url) => {
    if (!(0, exports.isResourceUrl)(url))
        return null;
    const match = url.match(resourceRegex);
    const itemId = match[2];
    let hash = match[3].trim();
    // In general we want the hash to be decoded so that non-alphabetical languages
    // appear as-is without being encoded with %.
    // Fixes https://github.com/laurent22/joplin/issues/1870
    if (hash)
        hash = (0, string_utils_1.urlDecode)(hash.substr(1)); // Remove the first #
    return {
        itemId: itemId,
        hash: hash,
    };
};
exports.parseResourceUrl = parseResourceUrl;
const fileUrlToResourceUrl = (fileUrl, resourceDir) => {
    let resourceDirUrl = (0, path_utils_1.toFileProtocolPath)(resourceDir);
    if (!resourceDirUrl.endsWith('/')) {
        resourceDirUrl += '/';
    }
    if (fileUrl.startsWith(resourceDirUrl)) {
        let result = fileUrl.substring(resourceDirUrl.length);
        // Remove the timestamp parameter, keep the hash.
        result = result.replace(/\?t=\d+(#.*)?$/, '$1');
        // Remove the file extension.
        result = result.replace(/\.[a-z0-9]+(#.*)?$/, '$1');
        result = `joplin://${result}`;
        if ((0, exports.isResourceUrl)(result)) {
            return result;
        }
    }
    return null;
};
exports.fileUrlToResourceUrl = fileUrlToResourceUrl;
const extractResourceUrls = (text) => {
    const markdownLinksRE = /\]\((.*?)\)/g;
    const output = [];
    let result = null;
    while ((result = markdownLinksRE.exec(text)) !== null) {
        const resourceUrlInfo = (0, exports.parseResourceUrl)(result[1]);
        if (resourceUrlInfo)
            output.push(resourceUrlInfo);
    }
    const htmlRegexes = [
        /<img[\s\S]*?src=["']:\/([a-zA-Z0-9]{32})["'][\s\S]*?>/gi,
        /<a[\s\S]*?href=["']:\/([a-zA-Z0-9]{32})["'][\s\S]*?>/gi,
    ];
    for (const htmlRegex of htmlRegexes) {
        while (true) {
            const m = htmlRegex.exec(text);
            if (!m)
                break;
            output.push({ itemId: m[1], hash: '' });
        }
    }
    return output;
};
exports.extractResourceUrls = extractResourceUrls;
const objectToQueryString = (query) => {
    if (!query)
        return '';
    let queryString = '';
    const s = [];
    for (const k in query) {
        if (!query.hasOwnProperty(k))
            continue;
        s.push(`${encodeURIComponent(k)}=${encodeURIComponent(query[k])}`);
    }
    queryString = s.join('&');
    return queryString;
};
exports.objectToQueryString = objectToQueryString;
//# sourceMappingURL=urlUtils.js.map